#' Plot summary chart
#'
#' Plot summary chart from a Lattes list object
#'
#' This function plots production information from a Lattes list object
#' generated by [lattes_to_list()]:
#' - Accepted journal papers
#' - Published journal papers
#' - Published conference papers
#' - Published book chapters
#' - Published books
#' - Ph.D. student defenses
#' - M.Sc. student defenses
#'
#' @param lattes.list a Lattes list object created using [lattes_to_list()]
#' @param chart.type package to use for generating the summary chart. "plotly" and
#' "rCharts" output interactive charts, "ggplot2" outputs a static one. Option
#' "rCharts" is currently disabled, until the rCharts package
#' ([https://github.com/ramnathv/rCharts](https://github.com/ramnathv/rCharts))
#' becomes available on CRAN.
#' @param width plot width (for "plotly" and "rCharts")
#' @param height plot height (for "plotly")
#' @param language Language to use in section headers
#' @param which.fields Character vector indicating which fields to include in
#'                     the productions page.
#'
#' @return plot object for inclusion in a productions page
#' (see [make_productions_page()].
#'
#' @export

plot_chart <- function(lattes.list,
                       chart.type   = c("ggplot2","plotly","rCharts"),
                       width        = 960,
                       height       = 480,
                       language     = c("EN", "PT"),
                       which.fields = c("journal.accepted",
                                        "journal.published",
                                        "conference.international",
                                        "conference.national",
                                        "book.chapters",
                                        "books",
                                        "phd.theses",
                                        "msc.theses")){

  # Match argument
  type <- match.arg(chart.type, c("ggplot2","plotly","rCharts"))

  # Check which.fields
  which.fields <- unique(tolower(which.fields))

  # preprocess lattes.list
  lattes.list$`Conference Papers - International` <- lattes.list$`Conference Papers`[which(lattes.list$`Conference Papers`$Internac), ]
  lattes.list$`Conference Papers - National`      <- lattes.list$`Conference Papers`[which(!lattes.list$`Conference Papers`$Internac), ]
  lattes.list$`Conference Papers`                 <- NULL

  # Select relevant fields
  ll <- lattes.list
  lattes.list <- vector(mode = "list", length = length(which.fields))
  for (i in seq_along(which.fields)){
    if (which.fields[i] == "books") {
      lattes.list[[i]] <- ll$Books
      names(lattes.list)[i] <- "Books"
    }
    if (which.fields[i] == "journal.accepted"){
      lattes.list[[i]] <- ll$`Accepted for Publication`
      names(lattes.list)[i] <- "Accepted for Publication"
    }
    if (which.fields[i] == "journal.published"){
      lattes.list[[i]] <- ll$`Journal Papers`
      names(lattes.list)[i] <- "Journal Papers"
    }
    if (which.fields[i] == "conference.international"){
      lattes.list[[i]] <- ll$`Conference Papers - International`
      names(lattes.list)[i] <- "Conference Papers - International"
    }
    if (which.fields[i] == "conference.national"){
      lattes.list[[i]] <- ll$`Conference Papers - National`
      names(lattes.list)[i] <- "Conference Papers - National"
    }
    if (which.fields[i] == "book.chapters"){
      lattes.list[[i]] <- ll$`Book Chapters`
      names(lattes.list)[i] <- "Book Chapters"
    }
    if (which.fields[i] == "phd.theses"){
      lattes.list[[i]] <- ll$`PhD Theses`
      names(lattes.list)[i] <- "PhD Theses"
    }
    if (which.fields[i] == "msc.theses"){
      lattes.list[[i]] <- ll$`MSc Dissertations`
      names(lattes.list)[i] <- "MSc Dissertations"
    }
  }

  # Assemble data frame with total counts per year
  myTotals              <- lapply(lattes.list,
                                  function(x){as.data.frame(table(x$Year))})
  myTotals              <- do.call(rbind.data.frame, myTotals)
  myTotals$Type         <- gsub("\\..*","", rownames(myTotals))
  names(myTotals)       <- c("Year", "Count", "Type")
  myTotals$Year         <- as.numeric(as.character(myTotals$Year))
  myTotals              <- myTotals[order(myTotals$Year), ]
  rownames(myTotals)    <- paste0(myTotals$Year, myTotals$Type)

  # Second data frame containing all entries by year (even if with 0 occurrences)
  myTotals2          <- cbind(expand.grid(Year = unique(myTotals$Year),
                                          Type = unique(myTotals$Type)),
                              Count = 0)
  rownames(myTotals2) <- paste0(myTotals2$Year, myTotals2$Type)

  # Populate counts for myTotals2
  for (i in 1:nrow(myTotals)){
    indx <- which(rownames(myTotals2) == rownames(myTotals)[i])
    if(length(indx)){
      myTotals2$Count[indx] <- myTotals$Count[i]
    }
  }


  ## Using ggplot2 (+ plotly if specified)
  if (type == "ggplot2" | type == "plotly"){
    if (language == "PT"){
      my.xlab <- "Ano"
      my.ylab <- "Quantidade"
      my.legtitle <- "Tipo"
    }
    if (language == "EN"){
      my.xlab <- "Year"
      my.ylab <- "Count"
      my.legtitle <- "Type"
    }

    myPlot <- ggplot2::ggplot(data    = myTotals2,
                              mapping = ggplot2::aes(x    = myTotals2$Year,
                                                     y    = myTotals2$Count,
                                                     fill = myTotals2$Type)) +
      ggplot2::geom_bar(stat     = "identity",
                        position = "stack",
                        colour   = "#00000011") +
      ggplot2::scale_x_continuous(breaks = min(myTotals2$Year):max(myTotals2$Year)) +
      ggplot2::theme(axis.text.x     = ggplot2::element_text(angle = 45, vjust = 0),
                     legend.position = "bottom") +
      ggplot2::xlab(my.xlab) +
      ggplot2::ylab(my.ylab) +
      ggplot2::labs(fill = my.legtitle)


    if (type == "plotly"){
      myPlot <- plotly::ggplotly(myPlot,
                                 width  = width,
                                 height = height)
    }
    return(myPlot)
  }

  if(type == "rCharts"){
    stop("rCharts option currently disabled. Use chart.type = 'plotly' instead.")
    #
    # --> Add rCharts and RColorBrewer as Imports in DESCRIPTION when available.
    # # Check if rCharts and RColorBrewer are installed
    # if(!("rCharts" %in% rownames(utils::installed.packages()))){
    #   stop("Please install rCharts 0.4.5 using\n> devtools::install_github(repo = 'ramnathv/rCharts', ref = '479a4f9')")
    # }
    # if(!("RColorBrewer" %in% rownames(utils::installed.packages()))){
    #   stop("Please install RColorBrewer using\n> install.packages('RColorBrewer')")
    # }
    #
    # # Using rCharts
    # TotalsPlot <- rCharts::nPlot(Count ~ Year,
    #                              group = "Type",
    #                              data  = myTotals2,
    #                              type  = "multiBarChart")
    # # Add decorations and customize style
    # TotalsPlot$chart(color            = RColorBrewer::brewer.pal(8, "Set1"),
    #                  reduceXTicks     = FALSE)
    # TotalsPlot$xAxis(rotateLabels     = -45,
    #                  staggerLabels    = FALSE)
    # TotalsPlot$chart(multibar.stacked = TRUE)
    # TotalsPlot$set(width = width)
    # TotalsPlot$print('iframesrc', include_assets = TRUE)
  }
}
