% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connectivity_estimation.multiple.R
\name{d.rel.conn.multiple}
\alias{d.rel.conn.multiple}
\alias{p.rel.conn.multiple}
\alias{q.rel.conn.multiple}
\alias{q.rel.conn.multiple.func}
\title{Functions for estimating the probability distribution of relative 
connectivity values as a weighted sum over possible input parameters}
\usage{
d.rel.conn.multiple(phi, ps, ks, ns, weights = 1,
  d.rel.conn = d.rel.conn.beta.prior, ...)

p.rel.conn.multiple(phi, ps, ks, ns, weights = 1,
  p.rel.conn = p.rel.conn.beta.prior, ...)

q.rel.conn.multiple.func(ps, ks, ns, weights = 1,
  p.rel.conn = p.rel.conn.beta.prior, N = 1000, ...)

q.rel.conn.multiple(q, ps, ks, ns, weights = 1,
  p.rel.conn = p.rel.conn.beta.prior, N = 1000, ...)
}
\arguments{
\item{phi}{Vector of fractions of individuals (i.e., eggs) from the source 
population settling at the destination population}

\item{ps}{Vector of fractions of individuals (i.e., eggs) marked in the 
source population}

\item{ks}{Vector of numbers of marked settlers found in sample}

\item{ns}{Vector of total numbers of settlers collected}

\item{weights}{Vector of weights for each set of p, k and n values}

\item{d.rel.conn}{Function to use to calculate probability density for 
individual combinations of \code{ps}, \code{ks} and \code{ns}. Defaults to 
\code{\link{d.rel.conn.beta.prior}}.  Could also be 
\code{\link{d.rel.conn.unif.prior}}.}

\item{p.rel.conn}{Function to use to calculate cumulative probability 
distribution for individual combinations of \code{ps}, \code{ks} and 
\code{ns}. Defaults to \code{\link{p.rel.conn.beta.prior}}. Could also be 
\code{\link{p.rel.conn.unif.prior}}.}

\item{N}{Number of points at which to estimate cumulative probability 
function for reverse approximation of quantile distribution. Defaults to 
\code{1000}.}

\item{q}{Vector of quantiles}

\item{\dots}{Additional arguments for the function \code{d.rel.conn} or
\code{p.rel.conn}}
}
\value{
Vector of probabilities or quantiles, or a function in the case of 
  \code{\link{q.rel.conn.multiple.func}}
}
\description{
These functions calculate the probability density function 
(\code{d.rel.conn.multiple}), the probability distribution function (aka the 
cumulative distribution function; \code{p.rel.conn.multiple}) and the 
quantile function (\code{q.rel.conn.multiple}) for the relative (to all 
settlers at the destination site) connectivity value for larval transport 
between a source and destination site. This version allows one to input 
multiple possible fractions of individuals (i.e., eggs) marked at the source 
site, multiple possible numbers of settlers collected and multiple possible 
marked individuals observed in the sample.  This gives one the possibility to
produce ensemble averages over different input parameter values with 
different probabilities of being correct.
}
\details{
If \code{ps}, \code{ks}, \code{ns} and \code{weights} can be scalars or 
vectors of the same length (or lengths divisible into that of the largest 
input parameter).  \code{weights} are normalized to sum to 1 before being 
used to sum probabilities from each individual set of input parameters.
}
\section{Functions}{
\itemize{
\item \code{d.rel.conn.multiple}: Estimates quantiles for the probability 
distribution function for relative connectivity between a pair of sites for
multiple possible \code{p}, \code{k} and \code{n} values.

\item \code{p.rel.conn.multiple}: Estimates the cumulative probability
distribution for relative connectivity between a paire of sites for
multiple possible \code{p}, \code{k} and \code{n} values.

\item \code{q.rel.conn.multiple.func}: Returns a function to estimate quantiles for
the probability distribution function for relative connectivity between a
pair of sites for multiple possible \code{p}, \code{k} and \code{n} values.

\item \code{q.rel.conn.multiple}: Estimates quantiles for the probability 
distribution function for relative connectivity between a pair of sites for
multiple possible \code{p}, \code{k} and \code{n} values.
}}
\examples{
library(ConnMatTools)

# p values have uniform probability between 0.1 and 0.4
p <- seq(0.1,0.8,length.out=100)

# Weights the same for all except first and last, which are halved
w <- rep(1,length(p))
w[1]<-0.5
w[length(w)]<-0.5

n <- 20 # Sample size
k <- 2 # Marked individuals in sample

# phi values to use for plotting distribution
phi <- seq(0,1,0.01)

prior.shape1 = 1 # Uniform prior
# prior.shape1 = 0.5 # Jeffreys prior

# Plot distribution
plot(phi,d.rel.conn.multiple(phi,p,k,n,w,prior.shape1=prior.shape1),
     main="Probability density for relative connectivity",
     xlab=expression(phi),
     ylab="Probability density",
     type="l")

# Add standard distributions for max and min p values
lines(phi,d.rel.conn.beta.prior(phi,min(p),k,n,prior.shape1=prior.shape1),
      col="red",lty="dashed")
lines(phi,d.rel.conn.beta.prior(phi,max(p),k,n,prior.shape1=prior.shape1),
      col="red",lty="dashed")

# Add some quantiles
q = q.rel.conn.multiple(c(0.025,0.25,0.5,0.75,0.975),
                        p,k,n,w,prior.shape1=prior.shape1)
abline(v=q,col="green")
}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\references{
Kaplan DM, Cuif M, Fauvelot C, Vigliola L, Nguyen-Huu T, Tiavouane J and Lett C 
  (in press) Uncertainty in empirical estimates of marine larval connectivity. 
  ICES Journal of Marine Science. doi:10.1093/icesjms/fsw182.
}
\seealso{
Other connectivity.estimation: \code{\link{d.rel.conn.beta.prior}},
  \code{\link{d.rel.conn.dists.func}},
  \code{\link{d.rel.conn.finite.settlement}},
  \code{\link{d.rel.conn.multinomial.unnorm}},
  \code{\link{d.rel.conn.unif.prior}},
  \code{\link{dual.mark.transmission}},
  \code{\link{optim.rel.conn.dists}},
  \code{\link{r.marked.egg.fraction}}
}

