% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trav_in_edge.R
\name{trav_in_edge}
\alias{trav_in_edge}
\title{Traverse from one or more selected nodes onto adjacent, inward edges}
\usage{
trav_in_edge(
  graph,
  conditions = NULL,
  copy_attrs_from = NULL,
  copy_attrs_as = NULL
)
}
\arguments{
\item{graph}{A graph object of class \code{dgr_graph}.}

\item{conditions}{An option to use filtering conditions for the traversal.}

\item{copy_attrs_from}{Providing a node attribute name will copy those node
attribute values to the traversed edges. If the edge attribute already
exists, the values will be merged to the traversed edges; otherwise, a new
edge attribute will be created.}

\item{copy_attrs_as}{If a node attribute name is provided in
\code{copy_attrs_from}, this option will allow the copied attribute values to be
written under a different edge attribute name. If the attribute name
provided in \code{copy_attrs_as} does not exist in the graph's edf, the new edge
attribute will be created with the chosen name.}
}
\value{
A graph object of class \code{dgr_graph}.
}
\description{
From a graph object of class \code{dgr_graph} move to incoming edges from a
selection of one or more selected nodes, thereby creating a selection of
edges. An optional filter by edge attribute can limit the set of edges
traversed to.
}
\details{
This traversal function makes use of an active selection of nodes. After the
traversal, depending on the traversal conditions, there will either be a
selection of edges or no selection at all.

Selections of nodes can be performed using the following node selection
(\verb{select_*()}) functions: \code{\link[=select_nodes]{select_nodes()}}, \code{\link[=select_last_nodes_created]{select_last_nodes_created()}},
\code{\link[=select_nodes_by_degree]{select_nodes_by_degree()}}, \code{\link[=select_nodes_by_id]{select_nodes_by_id()}}, or
\code{\link[=select_nodes_in_neighborhood]{select_nodes_in_neighborhood()}}.

Selections of nodes can also be performed using the following traversal
(\verb{trav_*()}) functions: \code{\link[=trav_out]{trav_out()}}, \code{\link[=trav_in]{trav_in()}}, \code{\link[=trav_both]{trav_both()}},
\code{\link[=trav_out_node]{trav_out_node()}}, \code{\link[=trav_in_node]{trav_in_node()}}, \code{\link[=trav_out_until]{trav_out_until()}}, or
\code{\link[=trav_in_until]{trav_in_until()}}.
}
\examples{
# Set a seed
suppressWarnings(RNGversion("3.5.0"))
set.seed(23)

# Create a simple graph
graph <-
  create_graph() \%>\%
  add_n_nodes(
    n = 2,
    type = "a",
    label = c("asd", "iekd")) \%>\%
  add_n_nodes(
    n = 3,
    type = "b",
    label = c("idj", "edl", "ohd")) \%>\%
  add_edges_w_string(
    edges = "1->2 1->3 2->4 2->5 3->5",
    rel = c(NA, "A", "B", "C", "D"))

# Create a data frame with node ID
# values representing the graph edges
# (with `from` and `to` columns), and,
# a set of numeric values
df <-
  data.frame(
    from = c(1, 1, 2, 2, 3),
    to = c(2, 3, 4, 5, 5),
    values = round(rnorm(5, 5), 2))

# Join the data frame to the graph's
# internal edge data frame (edf)
graph <-
  graph \%>\%
  join_edge_attrs(df = df)

# Show the graph's internal edge data frame
graph \%>\% get_edge_df()

# Perform a simple traversal from
# nodes to inbound edges with no
# conditions on the nodes
# traversed to
graph \%>\%
  select_nodes_by_id(nodes = 2) \%>\%
  trav_in_edge() \%>\%
  get_selection()

# Traverse from node `2` to any
# inbound edges, filtering to
# those edges that have NA values
# for the `rel` edge attribute
graph \%>\%
  select_nodes_by_id(nodes = 2) \%>\%
  trav_in_edge(
    conditions = is.na(rel)) \%>\%
  get_selection()

# Traverse from node `2` to any
# inbound edges, filtering to those
# edges that do not have NA values
# for the `rel` edge attribute
# (since there are no allowed
# traversals, the selection of node
# `2` is retained)
graph \%>\%
  select_nodes_by_id(nodes = 2) \%>\%
  trav_in_edge(
    conditions = !is.na(rel)) \%>\%
  get_selection()

# Traverse from node `5` to any
# inbound edges, filtering to those
# edges that have numeric values
# greater than `5.5` for the `rel`
# edge attribute
graph \%>\%
  select_nodes_by_id(nodes = 5) \%>\%
  trav_in_edge(
    conditions = values > 5.5) \%>\%
  get_selection()

# Traverse from node `5` to any
# inbound edges, filtering to those
# edges that have values equal to
# `D` for the `rel` edge attribute
graph \%>\%
  select_nodes_by_id(nodes = 5) \%>\%
  trav_in_edge(
    conditions = rel == "D") \%>\%
  get_selection()

# Traverse from node `5` to any
# inbound edges, filtering to those
# edges that have values in the set
# `C` and `D` for the `rel` edge
# attribute
graph \%>\%
  select_nodes_by_id(nodes = 5) \%>\%
  trav_in_edge(
    conditions = rel \%in\% c("C", "D")) \%>\%
  get_selection()

# Traverse from node `5` to any
# inbound edges, and use multiple
# conditions for the traversal
graph \%>\%
  select_nodes_by_id(nodes = 5) \%>\%
  trav_in_edge(
    conditions =
      rel \%in\% c("C", "D") &
      values > 5.5) \%>\%
  get_selection()

# Traverse from node `5` to any
# inbound edges, and use multiple
# conditions with a single-length
# vector
graph \%>\%
  select_nodes_by_id(nodes = 5) \%>\%
  trav_in_edge(
    conditions =
      rel \%in\% c("D", "E") |
      values > 5.5) \%>\%
  get_selection()

# Traverse from node `5` to any
# inbound edges, and use a regular
# expression as a filtering condition
graph \%>\%
  select_nodes_by_id(nodes = 5) \%>\%
  trav_in_edge(
    conditions = grepl("C|D", rel)) \%>\%
  get_selection()

# Show the graph's internal ndf
graph \%>\% get_node_df()

# Show the graph's internal edf
graph \%>\% get_edge_df()

# Perform a traversal from all
# nodes to their incoming edges and,
# while doing so, copy the `label`
# node attribute to any of the nodes'
# incoming edges
graph <-
  graph \%>\%
  select_nodes() \%>\%
  trav_in_edge(
    copy_attrs_from = label)

# Show the graph's internal edge
# data frame after this change
graph \%>\% get_edge_df()

}
