% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DsPA.R
\name{DsPA}
\alias{DsPA}
\title{Discrete power-Ailamujia distribution}
\usage{
DsPA(mu.link = "log", sigma.link = "logit")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "logit" link as the default for the sigma parameter. Other links are "probit" and "cloglog"'(complementary log-log).}
}
\value{
Returns a \code{gamlss.family} object which can be used
to fit a discrete power-Ailamujia distribution
in the \code{gamlss()} function.
}
\description{
The function \code{DsPA()} defines the
discrete power-Ailamujia distribution
a two parameter distribution,
for a \code{gamlss.family} object to be used in GAMLSS
fitting using the function \code{gamlss()}.
}
\details{
The discrete power-Ailamujia distribution with parameters \eqn{\mu} and \eqn{\sigma}
has a support 0, 1, 2, ... and density given by

\eqn{f(x | \mu, \sigma) = (\sigma^x)^\mu (1-x^\mu \log(\lambda)) -  (\sigma^{(x+1)})^\mu (1-(x+1)^\mu \log(\lambda))}

Note: in this implementation we changed the original parameters \eqn{\beta} and \eqn{\lambda}
for \eqn{\mu} and \eqn{\sigma} respectively, we did it to implement this distribution within gamlss framework.
}
\examples{
# Example 1
# Generating some random values with
# known mu and sigma
y <- rDsPA(n=100, mu=1.2, sigma=0.5)

# Fitting the model
library(gamlss)
mod1 <- gamlss(y~1, family=DsPA,
               control=gamlss.control(n.cyc=500, trace=TRUE))

# Extracting the fitted values for mu and sigma
# using the inverse link function
inv_logit <- function(x) 1/(1 + exp(-x))

exp(coef(mod1, what="mu"))
inv_logit(coef(mod1, what="sigma"))

# Example 2
# Generating random values under some model

# A function to simulate a data set with Y ~ DsPA
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  x3 <- runif(n)
  x4 <- runif(n)
  mu    <- exp(1 + 1.2*x1 + 0.2*x2)
  sigma <- inv_logit(2 + 1.5*x3 + 1.5*x4)
  y <- rDsPA(n=n, mu=mu, sigma=sigma)
  data.frame(y=y, x1=x1, x2=x2, x3=x3, x4=x4)
}

set.seed(123)
datos <- gendat(n=100)

mod2 <- gamlss(y~x1+x2, sigma.fo=~x3+x4, family=DsPA, data=datos,
               control=gamlss.control(n.cyc=500, trace=TRUE))

summary(mod2)

# Example 3
# failure times for a sample of 15 electronic components in an acceleration life test
# Taken from
# Alghamdi et. al (202) page 8354.

y <- c(1.0, 5.0, 6.0, 11.0, 12.0, 19.0, 20.0, 22.0,
       23.0, 31.0, 37.0, 46.0, 54.0, 60.0, 66.0)

mod3 <- gamlss(y~1, family=DsPA,
               control=gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu and sigma
# using the inverse link function
inv_logit <- function(x) 1/(1 + exp(-x))

exp(coef(mod3, what="mu"))
inv_logit(coef(mod3, what="sigma"))

# Example 4
# number of fires in Greece from July 1, 1998 to August 31, 1998.
# Taken from
# Alghamdi et. al (202) page 8354.

y <- c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1,
       1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2,
       2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3,
       3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4,
       4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5,
       5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6,
       6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7,
       8, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9,
       9, 9, 9, 9, 10, 10, 10, 11, 11,
       11, 11, 12, 12, 12, 12, 12, 12,
       15, 15, 15, 15, 16, 20, 43)

mod4 <- gamlss(y~1, family=DsPA,
               control=gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu and sigma
# using the inverse link function
inv_logit <- function(x) 1/(1 + exp(-x))

exp(coef(mod4, what="mu"))
inv_logit(coef(mod4, what="sigma"))

}
\references{
Alghamdi, A. S., Ahsan-ul-Haq, M., Babar, A., Aljohani, H. M., Afify, A. Z., & Cell, Q. E. (2022). The discrete power-Ailamujia distribution: properties, inference, and applications. AIMS Math, 7(5), 8344-8360.
}
\seealso{
\link{dDsPA}.
}
\author{
Maria Camila Mena Romana, \email{mamenar@unal.edu.co}
}
