\name{fac.design}
\alias{fac.design}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Function for full factorial designs }
\description{
  Function for creating full factorial designs with arbitrary numbers of levels, 
  and potentially with blocking
}
\usage{
fac.design(nlevels=NULL, nfactors=NULL, factor.names = NULL, 
        replications=1, repeat.only = FALSE, randomize=TRUE, seed=NULL, 
        blocks=1, block.gen=NULL, block.name="Blocks", bbreps=replications, 
        wbreps=1)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{nlevels}{ number(s) of levels, vector with \code{nfactors} entries or single number; 
                  can be omitted, if obvious from \code{factor.names} }
  \item{nfactors}{ number of factors, can be omitted if obvious from entries \code{nlevels} or \code{factor.names} }
  \item{factor.names}{ if \code{nlevels} is given, \code{factor.names} can be a character vector of factor names. 
        In this case, default factor levels are the numbers from 1 to the number of levels for each factor.\cr
        Otherwise it must be a list of vectors with factor levels. 
        If the list is named, list names represent factor names, otherwise default factor names are used.
        Default factor names are the first elements of the character vector \code{\link{Letters}}, 
        or the factors position numbers preceded by capital F in case of more than 50 factors.
        If both \code{nlevels} and factor.names are 
        given, they must be compatible.}
  \item{replications}{ positive integer number. Default 1 (i.e. each row just once). 
       If larger, each design run is executed replication times. 
       If \code{repeat.only}, repeated measurements 
       are carried out directly in sequence, i.e. no true replication takes place, 
       and all the repeat runs are conducted together. It is likely that the error 
       variation generated by such a procedure will be too small, so that average values 
       should be analyzed for an unreplicated design. 
       
       Otherwise (default), the full experiment is first carried out once, then 
       for the second replication and so forth. In case of randomization, 
       each such blocks is randomized separately. In this case, replication variance is 
       more likely suitable for usage as error variance 
       (unless e.g. the same parts are used for replication runs although build 
       variation is important).}
  \item{repeat.only}{ logical, relevant only if replications > 1. If \code{TRUE}, 
        replications of each run are grouped together 
       (repeated measurement rather than true replication). The default is 
       \code{repeat.only=FALSE}, i.e. the complete experiment 
       is conducted in \code{replications} blocks, and each run occurs in each block.  }
  \item{randomize}{ logical. If \code{TRUE}, the design is randomized. This is the default. 
       In case of replications, the nature of randomization depends on the setting of 
       option \code{repeat.only}.}
  \item{seed}{ optional seed for the randomization process (integer number)}
  \item{blocks}{ is the number of blocks into which the experiment is to be subdivided; \cr
       it must be a prime or a product of prime numbers which occur as common divisors 
       of the numbers of levels of several factors (cf. Details section).\cr
       
       If the experiment is randomized, randomization happens within blocks. \cr
       } 
  \item{block.gen}{ provides block generating information.\cr
       If \code{blocks} is a prime or a power of 2 (up to 2^8) or 3 (up to 3^5) 
       or a product of powers of 2, 3, and an individual other prime, 
       \code{block.gen} is not needed (but can be optionally specified).\cr
       If given, \code{block.gen} can be \cr
       a numeric vector of integer numbers \cr
       OR a numeric matrix with integer elements.
       
       There must be a row for each prime number into which \code{blocks} factorizes, 
       and a column for each (pseudo)factor into which the experimental 
       design factors can be partitioned (cf. Details and Examples sections and 
       function \code{\link{factorize}}).
       
       Rows for a p-level contributor to the block factor (p a prime) consist of 
       entries 0 to p-1 only.
       }
  \item{block.name}{ name of the block factor, default \dQuote{Blocks}}
  \item{bbreps}{ between block replications; these are always taken as genuine replications, 
       not repeat runs; default: equal to \code{replications};
       CAUTION: you should not modify \code{bbreps} if you do not work with blocks, 
       because the program code uses it instead of \code{replications} in some places }
  \item{wbreps}{ within block replications; whether or not these are taken as genuine replications 
       depends on the setting of \code{repeat.only}}
  }
\details{
   \code{fac.design} creates full factorial designs, i.e. the number of runs is the 
   product of all numbers of levels.\cr\cr
   It is possible to subdivide the design into blocks (one hierarchy level only) 
   by specifying an appropriate number of blocks. The method used is a generalization of 
   the one implemented in function \code{\link[conf.design]{conf.design}} for 
   symmetric factorials (i.e. factorials with all factors at the same prime number 
   of levels) and related to the method described in Collings (1984, 1989); 
   function \code{\link[conf.design]{conf.set}} from 
   package \pkg{conf.design} is used for checking the confounding 
   consequences of blocking. 
   
   Note that the number of blocks must 
   be compatible with the factor levels; it must factor into primes that occur with 
   high enough frequency among the pseudo-factors of the design. 
   This statement is now explained by an example: Consider a design with five factors at 
   2, 2, 3, 3, 6 levels. The 6-level factor can be thought of as consisting of 
   two pseudo-factors, a 2-level and a 3-level pseudo-factor, according to the 
   factorization of the number 6 into the two primes 2 and 3. It is possible \cr
   to obtain two blocks by confounding the two-factor interaction of the 
   two 2-level factors and the 2-level pseudo-factor of the 6-level factor, \cr
   or to obtain three blocks by confounding the blocking factor with the 
   three-factor interaction of the two three-level factors and the three-level 
   pseudo-factor of the 6-level factor, \cr
   or to get six blocks, by doing both simultaneously. \cr
   It is also possible to obtain 4 or 9 or even 36 blocks, if one is happy to 
   confound two-factor interactions with blocks. The 36 blocks are the product 
   of the 4 blocks from the 2-level portion with the nine blocks from the 3-level 
   portion. For each portion separately, there is a lookup-table for blocking 
   possibilities (\code{\link{block.catlg}}), for up to 128 blocks in 256 runs, 
   or up to 81 blocks in 243 runs. 
   
   5 blocks cannot be done for the above example design. Even if there were one additional 
   factor at 5 levels, it would still not be possible to do a number of blocks 
   with divisor 5, because this would confound the main effect of a factor with 
   blocks and would thus generate an error. 
   
   For any primes apart from 2 or 3, only one at a time can be handled automatically. 
   For example, if a design has three 5-level factors, it can be automatically subdivided 
   into 5 blocks by the option \code{blocks=5}. 
   It is also possible to run the design in 25 blocks; however, as 25=5*5, this cannot be done 
   automatically but has to be requested by specifying the \code{block.gen} option
   in addition to the \code{blocks} option 
   (in this case, \code{block.gen=rbind(c(1,0,1),c(1,1,0))} would do the job).
   
}
\value{
    \code{fac.design} returns a data frame of S3 class \code{\link{design}} 
    with attributes attached. 

   The experimental factors are all stored as R factors. \cr
   For factors with 2 levels, \code{\link{contr.FrF2}} contrasts (-1 / +1) are used. \cr
   For factors with more than 2 numerical levels, polynomial contrasts are used 
   (i.e. analyses will per default use orthogonal polynomials). \cr
   For factors with more than 2 categorical levels, the default contrasts are used.
   
   For changing the contrasts, use function \code{\link{change.contr}}.
   
   The \code{design.info} attribute of the data frame has the following elements:
  \describe{
    \item{type}{ character string \dQuote{full factorial} or 
         \dQuote{full factorial.blocked}}
    \item{nruns}{ number of runs (replications are not counted) }
    \item{nfactors}{ number of factors}
    \item{nlevels}{ vector with number of levels for each factor }
    \item{factor.names }{ list named with (treatment) factor names and containing as 
         entries vectors with coded factor levels }
    \item{nblocks}{for designs of type \code{full factorial.blocked} only; \cr
         number of blocks}
    \item{block.gen}{for designs of type \code{full factorial.blocked} only; \cr
         matrix the rows of which are the coefficients of the linear combinations 
         that create block columns from of pseudo factors}
    \item{blocksize}{for designs of type \code{full factorial.blocked} only; \cr
         size of each block (without consideration of \code{wbreps})}
    \item{replication }{ option setting in call to \code{FrF2} }
    \item{repeat.only }{ option setting in call to \code{FrF2} }
    \item{bbreps}{for designs of type \code{FrF2.blocked} only; 
         number of between block replications}
    \item{wbreps}{for designs of type \code{FrF2.blocked} only; 
         number of within block replications;\cr
         \code{repeat.only} indicates whether these are replications or repetitions only}
    \item{randomize }{ option setting in call to \code{FrF2} }
    \item{seed }{ option setting in call to \code{FrF2} }
    \item{creator }{ call to function FrF2 
         (or stored menu settings, if the function has been called via the R commander plugin 
         \pkg{RcmdrPlugin.DoE}) }
         }
}

\references{ 
  Collings, B.J. (1984). Generating the intrablock and interblock subgroups for 
     confounding in general factorial experiments. \emph{Annals of Statistics} \bold{12}, 
     1500--1509.

  Collings, B.J. (1989). Quick confounding. \emph{Technometrics} \bold{31}, 
     107--110.
  }
\author{ Ulrike Groemping }
\note{ This package is currently under intensive development. Substantial changes are to be expected in the near future.}
\seealso{ See also \code{\link[FrF2]{FrF2}}, \code{\link{oa.design}}, \code{\link[FrF2]{pb}}, \code{\link[conf.design]{conf.set}}, 
    \code{\link{block.catlg}}} 
\examples{
  ## only specify level combination 
  fac.design(nlevels=c(4,3,3,2))
  ## design requested via factor.names
  fac.design(factor.names=list(one=c("a","b","c"), two=c(125,275), 
     three=c("old","new"), four=c(-1,1), five=c("min","medium","max")))
  ## design requested via character factor.names and nlevels 
  ##    (with a little German lesson for one two three)
  fac.design(factor.names=c("eins","zwei","drei"),nlevels=c(2,3,2))
  
  ### blocking designs
  fac.design(nlevels=c(2,2,3,3,6), blocks=6, seed=12345)
  ## the same design, now unnecessarily constructed via option block.gen
  ## preparation: look at the numbers of levels of pseudo factors
  ## (in this order)
  unlist(factorize(c(2,2,3,3,6)))
  ## or, for more annotation, factorize the unblocked design
  factorize(fac.design(nlevels=c(2,2,3,3,6)))
  ## positions 1 2 5 are 2-level pseudo factors
  ## positions 3 4 6 are 4-level pseudo factors
  ## blocking with highest possible interactions
  G <- rbind(two=c(1,1,0,0,1,0),three=c(0,0,1,1,0,1))
  plan.6blocks <- fac.design(nlevels=c(2,2,3,3,6), blocks=6, block.gen=G, seed=12345)
  plan.6blocks
  
  ## two blocks, default design, but unnecessarily constructed via block.gen
  fac.design(nlevels=c(2,2,3,3,6), blocks=2, block.gen=c(1,1,0,0,1,0), seed=12345)
  
  ## three blocks, default design, but unnecessarily constructed via block.gen
  fac.design(nlevels=c(2,2,3,3,6), blocks=3, block.gen=c(0,0,1,1,0,1), seed=12345)
  
  ## nine blocks
  ## confounding two-factor interactions cannot be avoided
  ## there are warnings to that effect
  G <- rbind(CD=c(0,0,1,1,0,0),CE2=c(0,0,1,0,0,1))
  plan.9blocks <- fac.design(nlevels=c(2,2,3,3,6), blocks=9, block.gen=G, seed=12345)

  ## further automatic designs 
  fac.design(nlevels=c(2,2,3,3,6), blocks=4, seed=12345)
  fac.design(nlevels=c(2,2,3,3,6), blocks=9, seed=12345)
  fac.design(nlevels=c(2,2,3,3,6), blocks=36, seed=12345)
  fac.design(nlevels=c(3,5,6,10), blocks=15, seed=12345)
  
  ## independently check aliasing
  ## model with block main effects and all two-factor interactions
  ## 6 factors: not aliased
  summary(plan.6blocks)
  alias(lm(1:nrow(plan.6blocks)~Blocks+(A+B+C+D+E)^2,plan.6blocks))
  ## 9 factors: aliased
  summary(plan.9blocks)
  alias(lm(1:nrow(plan.9blocks)~Blocks+(A+B+C+D+E)^2,plan.9blocks))
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ array }
\keyword{ design }% __ONLY ONE__ keyword per line
