## ----setup, echo=FALSE---------------------------------------------------
library(knitr)
library(EEM)
opts_chunk$set(fig.width=6.5, fig.height=4)

## ----readEEM, eval=FALSE-------------------------------------------------
#  # choose folder interactively
#  data <- readEEM()
#  
#  # read raw data files from a folder
#  data <- readEEM(folder)
#  
#  # read raw data files from the current working folder
#  data <- readEEM(getwd())

## ----loadData------------------------------------------------------------
# load dataset
data(applejuice) 

# use summary to see information about the dataset.
summary(applejuice)

## ----drawEEM-------------------------------------------------------------
# draw EEM of sample no.1
drawEEM(applejuice, n = 1) 

# draw EEM of sample no.1 with different color
drawEEM(applejuice, n = 1, color.palette = cm.colors) 

## ----delScattering-------------------------------------------------------
# delete scattering regions and assign them as NA
applejuice_delS <- delScattering(applejuice, rep = NA) 
drawEEM(applejuice_delS, 1)

## ----delScattering2------------------------------------------------------
applejuice_delS <- delScattering(applejuice, rep = NA, first = 30, second = 0, third = 0, forth = 0) 
drawEEM(applejuice_delS, 1)

## ----delScattering3------------------------------------------------------
applejuice_delS <- delScattering(applejuice, rep = 0, 
                                 first = 30, second = 0, third = 0, forth = 0) 

## ----cutEEM--------------------------------------------------------------
applejuice_delS_cut <- cutEEM(applejuice_delS, cutEX = 350:500, cutEM = 500:700)
drawEEM(applejuice_delS_cut, 1)

## ----unfold--------------------------------------------------------------
## unfold EEM into EEM_uf (matrix form with samples x variables dimension)
applejuice_delS_uf <- unfold(applejuice_delS) 

# dimension of unfolded data
dim(applejuice_delS_uf)

# take a look at unfolded data
applejuice_delS_uf[1:5 ,1:5]

## ----normalize-----------------------------------------------------------
# normalize data
applejuice_delS_uf_norm <- normalize(applejuice_delS_uf) 

# the absolute sum of each row should equal to 1
rowSums(abs(applejuice_delS_uf_norm)) 

## ----pca-----------------------------------------------------------------
# perform PCA
result <- prcomp(applejuice_delS_uf_norm) # mean-centering is enabled by default

# plot scree plot
screeplot(result, npcs = 10, type = "lines", main = "Screeplot")

## ----scoreloading--------------------------------------------------------
# plot score plot 
plotScore(result, xPC = 1, yPC = 2) # pc 1 vs pc 2

# plot loading plot
plotLoading(result, ncomp = 1) # loading 1

## ----extractName---------------------------------------------------------
# extract sample name
sName <- names(applejuice) 

# country of apple production
country <- sapply(strsplit(sName, split = "-"), "[", 1) 
table(country) # counts of samples grouped by country

# cultivar of apples
cultivar <- sapply(strsplit(sName, split = "-"), "[", 2) 
table(cultivar) # counts of samples grouped by cultivar

## ----scoreg--------------------------------------------------------------
# plot score plot with grouping
plotScore(result, xPC = 1, yPC = 2,country, legendlocation = "topright")

# plot score using scatterplot matrix with grouping
plotScorem(result, ncomp = 5, country)
plotScorem(result, ncomp = 5, cultivar, pointsize = 1)

## ----pls, message=FALSE, warning=FALSE-----------------------------------
# load gluten data
data(gluten)
gluten_uf <- unfold(gluten) # unfold list into matrix

# delete columns with NA values
index <- colSums(is.na(gluten_uf)) == 0
gluten_uf <- gluten_uf[, index]
gluten_ratio <- as.numeric(names(gluten))

require(pls)
model <- plsr(gluten_ratio ~ gluten_uf, ncomp = 3)
plotLoading(model, ncomp = 3)
plotReg(model) 

