#' Apply the Exploratory Graph Analysis technique
#'
#' Estimates the number of dimensions of a given dataset/instrument
#' using graphical lasso (\code{\link{EBICglasso.qgraph}}) or the
#' Triangulated Maximally Filtered Graph (\code{\link[NetworkToolbox]{TMFG}})
#' method and the walktrap community detection algorithm (\code{\link[igraph]{cluster_walktrap}}).
#' The glasso regularization parameter is set via EBIC.
#'
#' @param data A dataframe with the variables to be used in the analysis or a correlation matrix.
#' If the data used is a correlation matrix, the argument \code{n} will need to be specified.
#'
#' @param n Integer.
#' Sample size, if the data provided is a correlation matrix
#'
#' @param plot.EGA Logical.
#' If TRUE, returns a plot of the network and its estimated dimensions.
#' Defaults to TRUE
#'
#' @param model A string indicating the method to use.
#' Current options are:
#'
#' \itemize{
#'
#' \item{\strong{\code{glasso}}}
#' {Estimates the Gaussian graphical model using graphical LASSO with
#' extended Bayesian information criterion to select optimal regularization parameter.
#' This is the default method}
#'
#' \item{\strong{\code{TMFG}}}
#' {Estimates a Triangulated Maximally Filtered Graph}
#'
#' }
#'
#' @param steps Number of steps to be used in \code{\link[igraph]{cluster_walktrap}} algorithm.
#' Defaults to 4.
#'
#' @param ... Additional arguments to be passed to \code{\link{EBICglasso.qgraph}}
#' or \code{\link[NetworkToolbox]{TMFG}}
#'
#' @author Hudson F. Golino <hfg9s at virginia.edu>
#'
#' @examples
#'
#' \donttest{
#' #estimate EGA
#' ega.wmt <- EGA(data = wmt2[,7:24], model = "glasso", plot.EGA = TRUE)
#'
#'
#' #estimate EGAtmfg
#' ega.wmt <- EGA(data = wmt2[,7:24], model = "TMFG", plot.EGA = TRUE)
#'
#' #summary statistics
#' summary(ega.wmt)
#'
#' #plot
#' plot(ega.wmt)
#'
#' #estimate EGA
#' ega.intel <- EGA(data = intelligenceBattery[,8:66], model = "glasso", plot.EGA = TRUE)
#'
#' #summary statistics
#' summary(ega.intel)
#'
#' #plot
#' plot(ega.intel)
#' }
#' @seealso \code{\link{bootEGA}} to investigate the stability of EGA's estimation via bootstrap
#' and \code{\link{CFA}} to verify the fit of the structure suggested by EGA using confirmatory factor analysis.
#'
#' @references
#' Golino, H. F., & Epskamp, S. (2017).
#' Exploratory graph analysis: A new approach for estimating the number of dimensions in psychological research.
#' \emph{PloS one}, \emph{12(6)}, e0174035..
#' doi: \href{https://doi.org/10.1371/journal.pone.0174035}{journal.pone.0174035}
#'
#' Golino, H. F., & Demetriou, A. (2017).
#' Estimating the dimensionality of intelligence like data using Exploratory Graph Analysis.
#' \emph{Intelligence}, \emph{62}, 54-70.
#' doi: \href{https://doi.org/10.1016/j.intell.2017.02.007}{j.intell.2017.02.007}
#'
#' Golino, H., Shi, D., Garrido, L. E., Christensen, A. P., Nieto, M. D., Sadana, R., & Thiyagarajan, J. A. (2018).
#' Investigating the performance of Exploratory Graph Analysis and traditional techniques to identify the number of latent factors: A simulation and tutorial.
#' \emph{PsyArXiv}.
#' doi: \href{https://psyarxiv.com/gzcre/}{10.31234/osf.io/gzcre}
#'
#' @importFrom stats cor
#'
#' @export
#'
# EGA default function - 11/21/2017
EGA <- function(data, n = NULL, model = c("glasso", "TMFG"), plot.EGA = TRUE, steps = 4, ...) {
  if(missing(model)){
    model = "glasso"
  }else{model = match.arg(model)}

  if(nrow(data)!=ncol(data)){
    data <- as.data.frame(data)
    if(model == "glasso"){
      cor.data <- qgraph::cor_auto(data)
      estimated.network <- EBICglasso.qgraph(S = cor.data, n = nrow(data), lambda.min.ratio = 0.1, returnAllResults = FALSE, ...)
    } else if(model == "TMFG"){
      cor.data <- NULL
      estimated.network <- NetworkToolbox::TMFG(data, normal = TRUE, na.data = "pairwise", ...)$A
    }

  } else if(nrow(data)==ncol(data)){
    cor.data <- data
    if(model == "glasso"){
      estimated.network <- EBICglasso.qgraph(S = data, n = n, lambda.min.ratio = 0.1, returnAllResults = FALSE, ...)
    } else if(model == "TMFG"){
      estimated.network <- NetworkToolbox::TMFG(cor.data, ...)$A
    }
  }

  graph <- NetworkToolbox::convert2igraph(abs(estimated.network))
  wc <- igraph::walktrap.community(graph, steps = steps)
  names(wc$membership) <- colnames(data)
  n.dim <- max(wc$membership)
  a <- list()
  a$n.dim <- n.dim
  a$correlation <- cor.data
  a$network <- estimated.network
  a$wc <- wc$membership
  dim.variables <- data.frame(items = colnames(data), dimension = a$wc)
  dim.variables <- dim.variables[order(dim.variables[, 2]), ]
  a$dim.variables <- dim.variables
  class(a) <- "EGA"
  if (plot.EGA == TRUE) {
    plot.ega <- qgraph::qgraph(estimated.network, layout = "spring",
                               vsize = 6, groups = as.factor(wc$membership),
                               label.prop = 1)
  }
  return(a)
}
