% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netdx.R
\name{netdx}
\alias{netdx}
\title{Dynamic Network Model Diagnostics}
\usage{
netdx(x, nsims = 1, dynamic = TRUE, nsteps, nwstats.formula = "formation",
  set.control.ergm, set.control.stergm, keep.tedgelist = FALSE,
  verbose = TRUE, ncores = 1)
}
\arguments{
\item{x}{An \code{EpiModel} object of class \code{netest}.}

\item{nsims}{Number of simulations to run.}

\item{dynamic}{If \code{TRUE}, runs dynamic diagnostics. If \code{FALSE} and
the \code{netest} object was fit with the Edges Dissolution approximation
method, simulates from the static ERGM fit.}

\item{nsteps}{Number of time steps per simulation (dynamic simulations only).}

\item{nwstats.formula}{A right-hand sided ERGM formula with the network
statistics of interest. The default is the formation formula of the
network model contained in \code{x}.}

\item{set.control.ergm}{Control arguments passed to simulate.ergm (see
details).}

\item{set.control.stergm}{Control arguments passed to simulate.stergm (see
details).}

\item{keep.tedgelist}{If \code{TRUE}, keep the timed edgelist generated from
the dynamic simulations, for further analysis on edge durations.}

\item{verbose}{Print progress to the console.}

\item{ncores}{Number of processor cores to run multiple simulations
on, using the \code{foreach} and \code{doParallel} implementations.}
}
\description{
Runs dynamic diagnostics on an ERGM/STERGM estimated through
             \code{netest}
}
\details{
The \code{netdx} function handles dynamic network diagnostics for network
models fit with the \code{netest} function. Given the fitted model, \code{netdx}
simulates a specified number of dynamic networks for a specified number of
time steps per simulation. The network statistics in \code{nwstats.formula}
are saved for each time step. Summary statistics for the formation model terms,
as well as dissolution model and relational duration statistics, are then
calculated for access when printing or plotting the \code{netdx} object.
}
\section{Control Arguments}{

Models fit with the full STERGM method in \code{netest} (setting \code{edapprox}
argument to \code{FALSE}) require only a call to \code{simulate.stergm}.
Control parameters for those simulations may be set using
\code{set.control.stergm} in \code{netdx}. The parameters should be input
through the \code{control.simulate.stergm()} function, with the available
parameters listed in the \code{\link{control.simulate.stergm}} help
page in the \code{tergm} package.

Models fit with the ERGM method with the edges dissolution approximation
(setting \code{edapprox} to \code{TRUE}) require a call first to
\code{simulate.ergm} for simulating an initial network and second to
\code{simulate.network} for simulating that static network forward through
time. Control parameters may be set for both processes in \code{netdx}.
For the first, the parameters should be input through the
\code{control.simulate.ergm()} function, with the available parameters listed
in the \code{\link[ergm:control.simulate.ergm]{control.simulate.ergm}} help
page in the \code{ergm} package. For the second, parameters should be input
through the \code{control.simulate.network()} function, with the available
parameters listed in the \code{\link{control.simulate.network}} help page in
the \code{tergm} package. An example is shown below.
}
\examples{
\dontrun{
# Network initialization and model parameterization
nw <- network.initialize(100, directed = FALSE)
formation <- ~edges
target.stats <- 50
coef.diss <- dissolution_coefs(dissolution = ~offset(edges), duration = 25)

# Estimate the model
est <- netest(nw, formation, target.stats, coef.diss, verbose = FALSE)

# Static diagnostics on the ERGM fit
dx1 <- netdx(est, nsims = 1e4, dynamic = FALSE,
             nwstats.formula = ~edges + meandeg + concurrent)
dx1
plot(dx1, method = "b", stats = c("edges", "concurrent"))

# Dynamic diagnostics on the STERGM approximation
dx2 <- netdx(est, nsims = 5, nsteps = 500,
             nwstats.formula = ~edges + meandeg + concurrent,
             set.control.ergm = control.simulate.ergm(MCMC.burnin = 1e6))
dx2
plot(dx2, stats = c("edges", "meandeg"), plots.joined = FALSE)
plot(dx2, type = "duration")
plot(dx2, type = "dissolution", method = "b", col = "bisque")
}

}
\seealso{
Plot these model diagnostics with \code{\link{plot.netdx}}.
}

