% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gemOLGFinancialInstrument.R
\name{gemOLGFinancialInstrument}
\alias{gemOLGFinancialInstrument}
\title{An Overlapping Generations Model with a Financial Instrument}
\usage{
gemOLGFinancialInstrument(...)
}
\arguments{
\item{...}{arguments to be passed to the function sdm2.}
}
\description{
Some examples of an overlapping generations model with a financial instrument (i.e. financial OLG model, FOLG model).
In these examples, there is a financial instrument (namely security) which serves as saving means and can be regarded as money, the shares of a firm, etc.
Consumers use this security for saving, and this is the only use of the security.
As Samuelson (1958) wrote, society by using money (i.e. security) will go from the non-optimal configuration
to the optimal configuration.

Assume that in each period the security will be split just like share split, and
the growth rate of the quantity of the security is equal to the growth rate of the population.
Obviously, this assumption will not affect the calculation results essentially.
And with this assumption, the equilibrium price vector can keep constant in each period, and
the nominal rates of profit and interest will equal the real rates of profit and interest (i.e. the population growth rate).
In contrast, in the time circle model the nominal rates of profit and interest equal zero and the real rates of profit and interest
equal the population growth rate.
}
\examples{
\donttest{
#### a FOLG economy with production and two-period-lived consumers
## See gemOLGPureExchange_2_2.
beta.firm <- c(1 / 3, 2 / 3)
# the population growth rate, the equilibrium interest rate and profit rate
GRExg <- 0.03

# the discount factor for the future instantaneous utility
time.discount.factor <- 0.9
# the discount factor for the future revenue
discount.factor <- time.discount.factor / (1 + GRExg)
saving.rate <- discount.factor / (1 + discount.factor)
beta.consumer <- c(1 - saving.rate, saving.rate)

dst.firm <- node_new(
  "prod",
  type = "CD", alpha = 5,
  beta = beta.firm,
  "lab", "prod"
)

dst.age1 <- node_new(
  "util",
  type = "CD", alpha = 1,
  beta = beta.consumer,
  "prod", "secy" # security, the financial instrument
)

dst.age2 <- node_new(
  "util",
  type = "Leontief", a = 1,
  "prod"
)

ge <- sdm2(
  A = list(
    dst.firm, dst.age1, dst.age2
  ),
  B = matrix(c(
    1, 0, 0,
    0, 0, 0,
    0, 0, 0
  ), 3, 3, TRUE),
  S0Exg = matrix(c(
    NA, NA, NA,
    NA, 1, NA,
    NA, NA, 1
  ), 3, 3, TRUE),
  names.commodity = c("prod", "lab", "secy"),
  names.agent = c("firm", "age1", "age2"),
  numeraire = "lab",
  GRExg = GRExg,
  maxIteration = 1,
  ts = TRUE
)

ge$p
matplot(ge$ts.p, type = "l")
matplot(diff(log(ge$ts.z)), type = "l") # log(1 + GRExg)
ge$D # the demand matrix of the current period
ge$S # the supply matrix of the current period
ge$S * (1 + GRExg) # the supply matrix of the next period

## the corresponding time-cycle model
n <- 5 # the number of periods, consumers and firms.
S <- matrix(NA, 2 * n, 2 * n)

S.lab.consumer <- diag((1 + GRExg)^(0:(n - 1)), n)
S[(n + 1):(2 * n), (n + 1):(2 * n)] <- S.lab.consumer

B <- matrix(0, 2 * n, 2 * n)
B[1:n, 1:n] <- diag(n)[, c(2:n, 1)]
B[1, n] <- (1 + GRExg)^(-n)

dstl.firm <- list()
for (k in 1:n) {
  dstl.firm[[k]] <- node_new(
    "prod",
    type = "CD", alpha = 5,
    beta = beta.firm,
    paste0("lab", k), paste0("prod", k)
  )
}

dstl.consumer <- list()
for (k in 1:(n - 1)) {
  dstl.consumer[[k]] <- node_new(
    "util",
    type = "CD", alpha = 1,
    beta = beta.consumer,
    paste0("prod", k), paste0("prod", k + 1)
  )
}

dstl.consumer[[n]] <- node_new(
  "util",
  type = "CD", alpha = 1,
  beta = beta.consumer,
  paste0("prod", n), "cc1"
)
node_set(dstl.consumer[[n]], "cc1",
  type = "Leontief", a = (1 + GRExg)^(-n), # a discounting factor
  "prod1"
)

ge2 <- sdm2(
  A = c(dstl.firm, dstl.consumer),
  B = B,
  S0Exg = S,
  names.commodity = c(paste0("prod", 1:n), paste0("lab", 1:n)),
  names.agent = c(paste0("firm", 1:n), paste0("consumer", 1:n)),
  numeraire = "lab1",
  policy = makePolicyMeanValue(30),
  maxIteration = 1,
  numberOfPeriods = 600,
  ts = TRUE
)

ge2$p
diff(log(ge2$p[1:n]))
diff(log(ge2$p[(n + 1):(2 * n)]))
ge2$D

#### an exchange economy with three-period-lived consumers (see Samuelson, 1958)
GRExg <- 0.03 # the population growth rate

dst.age1 <- node_new(
  "util",
  type = "CD", alpha = 1,
  beta = {
    saving.rate <- 1 / (1 + (1 + GRExg) + (1 + GRExg)^2)
    c(1 - saving.rate, saving.rate)
  },
  "cash", "secy"
)

dst.age2 <- node_new(
  "util",
  type = "CD", alpha = 1,
  beta = {
    saving.rate <- 1 / (1 + (1 + GRExg))
    c(1 - saving.rate, saving.rate)
  },
  "cash", "secy"
)

dst.age3 <- node_new(
  "util",
  type = "CD", alpha = 1, beta = c(1, 0),
  "cash", "secy"
)

ge <- sdm2(
  A = list(dst.age1, dst.age2, dst.age3),
  B = matrix(0, 2, 3),
  S0Exg = matrix(c(
    1 + GRExg, 1, 0,
    0, 0.5, 0.5
  ), 2, 3, TRUE),
  names.commodity = c("cash", "secy"),
  names.agent = c("age1", "age2", "age3"),
  numeraire = "cash",
  policy = function(time, state) {
    # Assume that unsold security will be void.
    last.Demand <- state$last.A \%*\% dg(state$last.z)
    secy.holding <- prop.table(last.Demand[2, ])
    if (time > 1) {
      state$S[2, 2] <- secy.holding[1]
      state$S[2, 3] <- secy.holding[2]
    }
    state
  }
)

ge$p
ge$S
ge$D
}

}
\references{
Samuelson, P. A. (1958) An Exact Consumption-Loan Model of Interest with or without the Social Contrivance of Money. Journal of Political Economy, vol. 66(6): 467-482.

de la Croix, David and Philippe Michel (2002, ISBN: 9780521001151) A Theory of Economic Growth: Dynamics and Policy in Overlapping Generations. Cambridge University Press.
}
\seealso{
{
\code{\link{gemOLGPureExchange_2_2}}
\code{\link{gemOLGTimeCircle}}
}
}
