#' @title Performing statistical inference after calibration
#' 
#' @description
#' \code{estimate} performs statistical inference after calibration.
#' 
#' @param formula An object of class "formula" specifying the calibration model. 
#' @param data An optional data frame containing the variables in the model (specified by \code{formula}).
#' @param calibration An object of class "calibration", generated by \code{GECalib}.
#' @param pimat An optional matrix contatining the joint inclusion probability matrix used for variance estimation.
#' 
#' @return A list of class \code{estimation} including the point estimates and its standard error. 
#' 
#' @references
#' Kwon, Y., Kim, J., & Qiu, Y. (2024). Debiased calibration estimation using generalized entropy in survey sampling.
#' Arxiv preprint <https://arxiv.org/abs/2404.01076>
#' 
#' Deville, J. C., and Särndal, C. E. (1992). Calibration estimators in survey sampling.
#' Journal of the American statistical Association, 87(418), 376-382.
#' 
#' @examples
#' set.seed(11)
#' N = 10000
#' x = data.frame(x1 = rnorm(N, 2, 1), x2= runif(N, 0, 4))
#' pi = pt((-x[,1] / 2 - x[,2] / 2), 3);
#' pi = ifelse(pi >.7, .7, pi)
#' 
#' delta = rbinom(N, 1, pi)
#' Index_S = (delta == 1)
#' pi_S = pi[Index_S]; d_S = 1 / pi_S
#' x_S = x[Index_S,,drop = FALSE]
#' # pimat = diag(d_S^2 - d_S) / N^2 # 1 / pi_i * (1 - 1 / pi_i)
#' 
#' e = rnorm(N, 0, 1)
#' y = x[,1] + x[,2] + e;
#' y_S = y[Index_S] # plot(x_S, y_S)
#' 
#' calibration0 <- GECal::GEcalib(~ 1, dweight = d_S, data = x_S,
#'                                const = N,
#'                                entropy = "SL", method = "DS")
#' GECal::estimate(y_S ~ 1, calibration = calibration0)$estimate # Hajek estimator
#' # sum(y_S * d_S) * N / sum(d_S)
#' 
#' calibration <- GECal::GEcalib(~ 0, dweight = d_S, data = x_S,
#' const = numeric(0),
#' entropy = "SL", method = "DS")
#' GECal::estimate(y_S ~ 1, calibration = calibration)$estimate # HT estimator
#' 
#' calibration1 <- GECal::GEcalib(~ ., dweight = d_S, data = x_S,
#'                                const = colSums(cbind(1, x)),
#'                                entropy = "ET", method = "DS")
#' GECal::estimate(y_S ~ 1, calibration = calibration1)$estimate
#' 
#' calibration2 <- GECal::GEcalib(~ ., dweight = d_S, data = x_S,
#'                                const = colSums(cbind(1, x)),
#'                                entropy = "ET", method = "GEC0")
#' GECal::estimate(y_S ~ 1, calibration = calibration2)$estimate
#' 
#' calibration3 <- GECal::GEcalib(~ . + g(d_S), dweight = d_S, data = x_S,
#'                                const = colSums(cbind(1, x, log(1 / pi))),
#'                                entropy = "ET", method = "GEC")
#' GECal::estimate(y_S ~ 1, calibration = calibration3)$estimate
#' 
#' calibration4 <- GECal::GEcalib(~ . + g(d_S), dweight = d_S, data = x_S,
#'                                const = colSums(cbind(1, x, NA)),
#'                                entropy = "ET", method = "GEC")
#' GECal::estimate(y_S ~ 1, calibration = calibration4)$estimate
#' 
#' calibration5 <- GECal::GEcalib(~ . + g(d_S), dweight = d_S, data = x_S,
#'                                const = colSums(cbind(1, x, NA)),
#'                                entropy = "ET", method = "GEC", K_alpha = "log")
#' GECal::estimate(y_S ~ 1, calibration = calibration5)$estimate
#' 
#' @export
estimate <- function(formula, data = NULL, calibration, pimat = NULL){
  
  response_vars <- all.vars(formula[[2]])
  
  environment(formula) <- environment()
  
  if (is.null(data)) {
    mf <- model.frame(formula, parent.frame())  # Evaluate in parent environment
  } else {
    mf <- model.frame(formula, data)  # Evaluate in the provided data
  }
  
  ys <- as.matrix.data.frame(mf[,response_vars, drop = FALSE])
  
  # # If data is provided, extract the variables from the data
  # if (!is.null(data)) {
  #   # Extract the columns from the data corresponding to the response variables
  #   ys <- do.call(cbind, lapply(response_vars, function(var) data[[var]]))
  # } else {
  #   # If no data is provided, evaluate the variables in the global environment
  #   ys <- do.call(cbind, lapply(response_vars, function(var) eval(parse(text = var), envir = parent.frame())))
  # }
  
  if (!inherits(calibration, "calibration")){
    stop("Object calibration is not a class \"calibration\".")
  }
  
  Xs <- calibration$Xs
  w <- calibration$w
  dweight <- calibration$dweight
  entropy <- calibration$entropy
  del <- calibration$del
  G.scale <- calibration$G.scale
  weight.scale <- calibration$weight.scale
  method <- calibration$method
  const <- calibration$const
  What <- calibration$What
  K_alpha <- calibration$K_alpha
  col_position <- calibration$col_position
  
  if(is.null(pimat)) pimat = diag(w * (w - 1)) # dweight * (dweight - 1)
  
  if(length(const) == 0){
    yhat = rep(0, length(ys))
  }else{
    if(method == "DS"){
      gammahat = solve(t(Xs) %*% (Xs * dweight / G.scale),
                       t(Xs) %*% (ys * dweight / G.scale))
    }else if(method == "GEC" | method == "GEC0"){
      gammahat = solve(t(Xs) %*% (Xs * fprime(dweight * weight.scale, entropy = entropy, 
                                              del = del) / G.scale / weight.scale^2),
                       t(Xs) %*% (ys * fprime(dweight * weight.scale, entropy = entropy, 
                                              del = del) / G.scale / weight.scale^2))
    }
    
    if(method == "GEC" & any(is.na(const))){
      hatSigmazz = t(Xs) %*% (Xs * fprime(dweight * weight.scale, entropy = entropy, 
                                          del = del) / G.scale / weight.scale^2)
      xcolums = sequence(ncol(Xs))[-col_position]; gcolums = col_position
      hatSigmaxx = hatSigmazz[xcolums, xcolums]
      hatSigmagx = hatSigmazz[gcolums, xcolums, drop = FALSE]
      hatSigmagg = hatSigmazz[gcolums, gcolums, drop = FALSE]
      hatSigmagg_x = drop(hatSigmagg - hatSigmagx %*% solve(hatSigmaxx, t(hatSigmagx)))
      
      if(identical(K_alpha, identity)){
        Xs[,gcolums] <- c(hatSigmagx %*% solve(hatSigmaxx, t(Xs[,xcolums])))
      }else{
        N = const[1]
        Xs[,gcolums] <- (1 / hatSigmagg_x / (1 / (What + N) + 1 / hatSigmagg_x)) * c(hatSigmagx %*% solve(hatSigmaxx, t(Xs[,xcolums])))
      }
    }
    yhat = drop(Xs %*% gammahat)
  }
  


  Varhat = drop(t(ys - yhat) %*% pimat %*% (ys - yhat))

  res_list <- list(cov = Varhat,
                   estimate = cbind("Estimate" = colSums(ys * w), "Std. Error" = drop(sqrt(diag(Varhat, nrow = ncol(ys))))))
  
  class(res_list) <- "estimation"
  return(res_list)
}
