% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.GP.R
\name{plot.GP}
\alias{plot.GP}
\title{Plotting GP model fits}
\usage{
\method{plot}{GP}(x, M = 1, range = c(0, 1), resolution = 50,
  colors = c("black", "blue", "red"), line_type = c(1, 2), pch = 20,
  cex = 1, legends = FALSE, surf_check = FALSE, response = TRUE,
  ...)
}
\arguments{
\item{x}{a class \code{GP} object estimated by \code{GP_fit}}

\item{M}{the number of iterations for use in prediction. See
\code{\link{predict.GP}}}

\item{range}{the input range for plotting (default set to \code{[0, 1]})}

\item{resolution}{the number of points along a coordinate in the specified
\code{range}}

\item{colors}{a vector of length 3 assigning \code{colors[1]} to training
design points, \code{colors[2]} to model predictions, and \code{colors[3]}
to the error bounds}

\item{line_type}{a vector of length 2 assigning \code{line_type[1]} to model
predictions, and \code{line_type[2]} to the error bounds}

\item{pch}{a parameter defining the plotting character for the training
design points, see `pch' for possible options in \code{\link{par}}}

\item{cex}{a parameter defining the size of the \code{pch} used for plotting
the training design points, see `cex' for possible options in
\code{\link{par}}}

\item{legends}{a parameter that controls the inclusion of a
\code{\link{legend}}; by default it is `FALSE'}

\item{surf_check}{logical, switch between 3d surface and 2d level/contour
plotting, the default of \code{FALSE} implies level/contour plotting}

\item{response}{logical, switch between predicted response and error (MSE)
plots, the default of \code{TRUE} displays the response surface}

\item{\dots}{additional arguments from \code{\link{wireframe}} or
\code{\link{levelplot}}}
}
\description{
Plots the predicted response and mean squared error (MSE) surfaces for
simulators with 1 and 2 dimensional inputs (i.e. d = 1,2).
}
\examples{

## 1D Example 1
n <- 5
d <- 1 
computer_simulator <- function(x){
    x <- 2 * x + 0.5
    y <- sin(10 * pi * x) / (2 * x) + (x - 1)^4
    return(y)
}
set.seed(3)
library(lhs)
x <- maximinLHS(n,d)
y <- computer_simulator(x)
GPmodel <- GP_fit(x,y)
plot(GPmodel)


## 1D Example 2
n <- 7
d <- 1
computer_simulator <- function(x) {
    y <- log(x + 0.1) + sin(5 * pi * x)
    return(y)
}
set.seed(1)
library(lhs)
x <- maximinLHS(n,d)
y <- computer_simulator(x)
GPmodel <- GP_fit(x, y)
## Plotting with changes from the default line type and characters
plot(GPmodel, resolution = 100, line_type = c(6,2), pch = 5)


## 2D Example: GoldPrice Function
computer_simulator <- function(x) {
    x1 <- 4 * x[, 1] - 2
    x2 <- 4 * x[, 2] - 2
    t1 <- 1 + (x1 + x2 + 1)^2 * (19 - 14 * x1 + 3 * x1^2 - 14 * x2 + 
        6 * x1 * x2 + 3 * x2^2)
    t2 <- 30 + (2 * x1 - 3 * x2)^2 * (18 - 32 * x1 + 12 * x1^2 + 48 * x2 - 
        36 * x1 * x2 + 27 * x2^2)
    y <- t1 * t2
    return(y)
}
n <- 30 
d <- 2
set.seed(1)
x <- lhs::maximinLHS(n, d) 
y <- computer_simulator(x)
GPmodel <- GP_fit(x, y)
## Basic level plot
plot(GPmodel)
## Adding Contours and increasing the number of levels
plot(GPmodel, contour = TRUE, cuts = 50, pretty = TRUE)
## Plotting the Response Surface
plot(GPmodel, surf_check = TRUE)
## Plotting the Error Surface with color
plot(GPmodel, surf_check = TRUE, response = FALSE, shade = TRUE)

}
\references{
Ranjan, P., Haynes, R., and Karsten, R. (2011). A
Computationally Stable Approach to Gaussian Process Interpolation of
Deterministic Computer Simulation Data, Technometrics, 53(4), 366 - 378.
}
\seealso{
\code{\link{GP_fit}} for estimating the parameters of the GP model;
\cr \code{\link{predict.GP}} for predicting the response and error surfaces;
\cr \code{\link{par}} for additional plotting characters and line types for
1 dimensional plots; \cr \code{\link{wireframe}} and \code{\link{levelplot}}
for additional plotting settings in 2 dimensions.
}
\author{
Blake MacDonald, Hugh Chipman, Pritam Ranjan
}
