% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/common_equivalence.R
\name{equivalence}
\alias{equivalence}
\title{Checking optimality of a design with respect to locally D-optimal, minimax D-optimal and standardized maximin D-optimal criteria}
\usage{
equivalence(fimfunc, x, w, lx, ux, lp, up, type, n.seg = 6,
  maxeval_equivalence = 6000, locally = NULL, control_gosolnp = NULL,
  plot_3d = "lattice", plot_sensitivity = TRUE, ...)
}
\arguments{
\item{fimfunc}{Fisher information matrix. Can be the name of the Fisher information matrix from FIM family functions available in this package as a
character string or a function that returns the information matrix. See "Details" of \code{\link{mica}}.}

\item{x}{a vector of design points. If the  model has \eqn{n} explanatory variables, let \eqn{x_{ij}}
 be the \eqn{j}th component of the $\eqn{i}th design point.
The argument \code{x} is \eqn{x = (x_{11}, x_{21},..., x_{k1},..., x_{1n}, x_{2n},... x_{kn})}.
See "Examples" on how to set this argument when the design space does not have one dimension, e.g. is of two-dimension.}

\item{w}{a vector of design weights}

\item{lx}{lower bound of the design space \eqn{\chi}}

\item{ux}{upper bound of the design space \eqn{\chi}}

\item{lp}{lower bound of the region of uncertainty \eqn{\Theta}. As same order as the argument \code{param} of \code{fimfunc}.}

\item{up}{upper bound of the region of uncertainty \eqn{\Theta}.
For  the locally D-optimal design,\code{lp} and \code{up}  must be fixed to the same values, i.e. \code{lp = up}.}

\item{type}{a character string that shows the type of optimal design. \code{"minimax"} for a minimax optimal design, \code{"standardized"} for a standardized maximin D-optimal design and \code{"locally"} for a locally D-optimal design.
When \code{type = "locally"}, then  \code{lp} must be set equal to \code{up}.}

\item{n.seg}{number of starting points to find all local optima of the inner problem for constructing the answering set
\eqn{A(\xi)} or \eqn{N(\xi)} is equal to \code{(n.seg + 1)^p}. Applicable only when
 \code{type = "standardized"} or \code{type = "minimax"}. Defaults to \code{4}. See "Details".}

\item{maxeval_equivalence}{maximum number of evaluations (\code{maxeval})  that will be passed to optimization
function \code{\link[nloptr]{directL}} to find the maximum of the sensitivity function required for calculating ELB. See "Details".}

\item{locally}{a function that returns the value of determinant of FIM for
the locally D-optimal design, i.e.  \eqn{|M(\xi_{\bold{\theta}}, \bold{\theta})|}{|M(\xi_\theta, \theta)|}.
 Only required when \code{type = "standardized"}. See "Details".}

\item{control_gosolnp}{tuning parameters of the function \code{\link[Rsolnp]{gosolnp}}.
Only when 1) \code{type = "standardized"} 2) \code{fimfunc} is a character string  3) locally D-optimal design is not available in a closed-form.
See "Details" for the components.}

\item{plot_3d}{a character strings shows which plotting package should be used to plot the two-dimensional sensitivity function.
\code{plot_3d = "lattice"} to use package \link[lattice]{lattice} and \code{plot_3d = "rgl"} to use package \link[rgl]{rgl}. The package should be installed before use.}

\item{plot_sensitivity}{\code{logical}, if \code{TRUE}, the sensitivity function will be plotted.}

\item{...}{further argument to be passed to \code{fimfunc}.}
}
\value{
an object of class \code{'equivalence'} that is a list contains:
 \describe{
 \item{\code{type}}{argument \code{type} that is required for print methods.}
 \item{\code{all_optima}}{a \code{matrix}; all the local maxima of  the minimax D-criterion for minimax optimal design problems
   or all the local minima of the standardized maximin D-criterion standardized maximin optimal design problems.
    Each row is one element of a set of parameters.
      For locally optimal designs it will be set to \code{NA}.}
 \item{\code{all_optima_cost}}{cost of each element of \code{all_optima}. \code{NA} for locally optimal designs.}
 \item{\code{answering}}{a matrix; answering set \eqn{A(\xi)} or \eqn{N(\xi)}  chosen from \code{all_optima}
  based on a pre-chosen tolerance, say \eqn{0.005}. Each row is one element of the set. \code{NA} for locally optimal designs.}
 \item{\code{answering_cost}}{cost of each element of answering set. \code{NA} for locally optimal designs.}
 \item{\code{mu}}{the probability measure on the answering set. It is \code{NA} for a locally optimal design.}
 \item{\code{max_deriv}}{maximum of the sensitivity function}
 \item{\code{ELB}}{D-efficiency lower bound. If negative, probably a local maximum has been found. In this case,
 the value of \code{maxeval_equivalence} should be increased to find the global maximum.}
 \item{\code{crtval}}{criterion value.  In minimax optimal design it is equal to the minimum of \code{answering_cost} and for the
   standardized maximin optimal design it is equal to the maximum of \code{answering_cost}.
    For the locally D-optimal design it is the value of the logarithm of the determinant of the FIM. See "Details".}
 }
}
\description{
An approximate design \eqn{\xi} is a probability measure defined on a user-selected design space \eqn{\chi}.
Let \eqn{\Xi} be the space of all such designs on \eqn{\chi} and let \eqn{\xi}
 be an approximate design with \eqn{k} support points  at \eqn{\boldsymbol{x}_1, \boldsymbol{x}_2, ..., \boldsymbol{x}_k}{x1, x2, ..., xk}
  from \eqn{\chi} with corresponding weights  \eqn{w_1, . . . ,w_k}{w1, ..., wk},
  \eqn{\sum_{i=1}^{k} w_i = 1}{w1 + w2 + ...+ wk = 1}.
   A design \eqn{\xi^*}{\xi*} is minimax D-optimal among all designs on \eqn{\chi} if and only if there exists a probability measure \eqn{\mu^*}{\mu*} on
   \deqn{A(\xi^*) = \left\{\nu \in \Theta \mid -log|M(\xi^*, \nu)| = \max_{\theta \in \Theta} -log|M(\xi^*, \theta)| \right\},}{
     A(\xi*) = {\nu belongs to \Theta | -log|M(\xi*, \nu)| = maxization with respect to \theta over \Theta on function -log|M(\xi*, \theta)|} ,}
       such that the following inequality holds for all \eqn{\boldsymbol{x} \in \chi}{x belong to \chi}
        \deqn{c(\boldsymbol{x}, \mu^*, \xi^*) = \int_{A(\xi^*)} tr M^{-1}(\xi^*, \nu)I(\boldsymbol{x}, \nu)\mu^* d(\nu)-p \leq 0,}{
         c(x, \mu*, \xi*) = integration over A(\xi*) with integrand  tr M^-1(\xi*, \nu)I(x, \nu)\mu* d(\nu)-p <= 0,}
          with equality at all support points of \eqn{\xi^*}{\xi*}.
           Here, \eqn{p} is the number of model parameters. \eqn{c(\boldsymbol{x}, \mu^*, \xi^*)}{c(x, \mu*, \xi*)} is called \bold{sensitivity function}.
The set \eqn{A(\xi^*)}{A(\xi*)} is sometimes called the \bold{answering set} of
 \eqn{\xi^*}{\xi*} and the measure \eqn{\mu^*}{\mu*} is a subgradient of the
   non-differentiable criterion evaluated at \eqn{M(\xi^*,\nu)}{M(\xi*,\nu)}.\cr
For standardized maximin D-optimal design the answering set \eqn{N(\xi^*)}{N(\xi*)} is
   \deqn{N(\xi^*) = \left\{\boldsymbol{\nu} \in \Theta \mid \mbox{eff}_D(\xi^*, \boldsymbol{\nu}) = \min_{\boldsymbol{\theta} \in \Theta} \mbox{eff}_D(\xi^*, \boldsymbol{\theta}) \right\}.
     }{N(\xi*) = \{\nu belongs to \Theta  |eff_D(\xi*, \nu) = minimum over \Theta eff_D(\xi*, \theta) \},} where
     \eqn{\mbox{eff}_D(\xi, \boldsymbol{\theta}) =  (\frac{|M(\xi, \boldsymbol{\theta})|}{|M(\xi_{\boldsymbol{\theta}}, \boldsymbol{\theta})|})^\frac{1}{p}}{
     eff_D(\xi, \theta) =  (|M(\xi, \theta)|/|M(\xi_\theta, \theta)|)^(1/p)} and \eqn{\xi_\theta} is the locally D-optimal design with respect to \eqn{\theta}. \cr
For locally optimal designs the answering set has only one element that is \eqn{\nu = \boldsymbol{\theta_0} }{\nu = \theta_0} and \eqn{\mu =1}.
}
\details{
There is no theoretical rule on how to choose the number of points in \eqn{A(\xi^*)}{A(\xi*)} as support
 for the measure \eqn{\mu^*}{\mu*}  and they would have to be found by trial and error.
  To this end, we first find all the local maxima for optimization over \eqn{\Theta} by a local search (L-BFGS-B) with
   different \code{(n.seg + 1)^p} starting points  and then pick the ones nearest to the global minimum
    subject to a tolerance of \eqn{0.005}.\cr


We measure the closeness of a design
 \eqn{\xi} to the minimax optimal design  using its minimax D-efficiency defined by
  \deqn{d_{\mbox{eff}} = \left(\frac{\max_{\boldsymbol{\theta} \in \Theta} -\log|M(\xi_D, \boldsymbol{\theta})|}{\max_{\boldsymbol{\theta} \in \Theta} -\log|M(\xi, \boldsymbol{\theta})|}\right)^{1/p},}{
    d_eff = {(maximum over \Theta -log|M(\xi_D, \theta)|)/(maximum over \Theta -log|M(\xi, \theta)|)}^(1/p),}
      where \eqn{\xi_D} is  the minimax D-optimal design.
We may obtain a D-efficiency Lower Bound (ELB) for the minimax D-efficiency of a design \eqn{\xi} without knowing \eqn{\xi^*}{\xi*}.
ELB is calculated by \eqn{p/(p + max_{\boldsymbol{x} \in \chi}c(\boldsymbol{x}, \mu, \xi))}{p/(p + maximum over \chi c(x, \mu, \xi))}, where \eqn{\mu^*}{\mu*}
    is the probability measure defined on \eqn{A(\xi)} that maximizes \eqn{c(x_\xi,\mu,\xi)}{c(x_\xi,\mu,\xi)} over all probability measures \eqn{\mu}
     and \eqn{x_\xi}{x_\xi} is any arbitrary support point of \eqn{\xi}.\cr
Using similar arguments, we can find the D-efficiency lower bound for the standardized maximin D-optimal and locally D-optimal designs.


 For standardized maximin D-optimal designs, if \code{fimfunc} is a \bold{character} strings from the list of available FIM, then the function \code{locally} is created automatically as follows:
  \itemize{
     \item{ if a \bold{closed-form} is available for the locally D-optimal designs for the model of interest: }{
       function \code{locally} combines  some algebraic functions to return the value of the determinant. See "Details"
        of each defined FIM for the formula.}
     \item{if  \bold{no closed-form} is available for the locally D-optimal designs for the model of interest: }{
        function \code{\link[Rsolnp]{gosolnp}} is used to find the locally D-optimal design
         (within the class of minimally-supported and equally-weighted designs).
          Argument \code{control_gosolnp} is a list that contains some tuning parameters of the function \code{gosolnp} that are  \code{n.sim} (default \code{500}), \code{n.restarts} (default \code{1}) and \code{trace} (default \code{FALSE}).}
}

We note that if \code{fimfunc} is defined by the user, defining \code{locally} is  obligatory.
In this case function \code{locally} must have two arguments: 1) model parameters \code{param} 2)  \code{auxiliary} that is for internal use.\cr

To calculate the ELB, it is necessary to find the global maximum of the sensitivity function.
if a local maximum is found then a negative value for ELB may be reported.
In this case, increasing the value of \code{maxeval_equivalence} can fix the computational issue.\cr

 The criterion value for locally D-optimal design is
 \deqn{-\log|M(\xi, \boldsymbol{\theta_0} )|;}{-log|M(\xi, \theta_0 )|;}
  for minimax optimal design is (global maximum)
 \deqn{\max_{\theta \in \Theta} -\log|M(\xi, \theta)|;}{max -log|M(\xi, \theta)|;}
 for standardized maximin optimal design is (global minimum)
 \deqn{\inf_{\theta \in \Theta} \left[\left(\frac{|M(\xi, \theta)|}{|M(\xi_{\theta}, \theta)|}\right)^\frac{1}{p}\right].}{
  inf {|M(\xi, \theta)| / |M(\xi_\theta, \theta)|}^p.}
}
\examples{
#############################################################
## check locally optimality: lp = up and type = "locally"
inipar <- c(2, 3)
equivalence (fimfunc = "FIM_logistic",
            x = c(1.485526, 2.51446 ),
            w = c(.5, .5),
            lx = -5, ux = 5,
            lp = inipar, up = inipar,
            type = "locally")

##############################################################################
## standardized maximin D-optimal design does not depend on theta0 and theta1,
##  so we fix them locally D-optimal design has a closed-form which is defined
##  internally
equivalence (fimfunc = "FIM_loglin",
            x = c(0, 4.2494, 17.0324, 149.9090),
            w = c(0.3204, 0.1207, 0.2293, 0.3296),
            lx = 0, ux = 150,
            lp = c(2, 2, 1), up = c(2, 2, 15),
            type = "standardized")

##############################################################################
\dontrun{
## there is no analytical solution for locally optimal design for this model
## gosolnp automatically will be used to find the locally
##  optimal design in the denominator of standardized criterion.
##  Becasue, it is two-level nested optimization
##  (first level on parameter space) and second level on design space)
##  it takes so long to find 'all_optima' and construct 'answerign' set.
equivalence (fimfunc = "FIM_power_logistic",
            x = c(-4.5515, 0.2130, 2.8075),
            w = c(0.4100, 0.3723, 0.2177),
            lx = -5, ux = 5,
            lp = c(0, 1), up = c(3, 1.5),
            type = "standardized",
            s = .2)
}


############################################################################
### when a design point is of two dimension
\dontrun{
equivalence (fimfunc = "FIM_mixed_inhibition",
            x = c(3.4614, 4.2801, 30, 30, 0, 3.1426, 0, 4.0373 ),
            w = rep(1/4, 4),
            lx = c(0, 0), ux = c(30, 60),
            lp = c(7, 4, 2, 4), up = c(7, 5, 3, 5),
            type = "standardized")
## here the design points are x1 = c(3.4614, 0), x2 = c(4.2801, 3.1426),
## x3 = c(30, 0), x4 = c(30, 4.0373)
## using package rgl (rgl must be installed for plot)
equivalence (fimfunc = "FIM_mixed_inhibition",
            x = c(3.4614, 4.2801, 30, 30, 0, 3.1426, 0, 4.0373 ),
            w = rep(1/4, 4),
            lx = c(0, 0), ux = c(30, 60),
            lp = c(7, 4, 2, 4), up = c(7, 5, 3, 5),
            type = "standardized", plot_3d = "rgl")

equivalence (fimfunc = "FIM_comp_inhibition",
            x = c(3.4432, 30, 30, 0, 0, 18.8954),
            w = rep(1/3, 3),
            lx = c(0, 0), ux = c(30, 60),
            lp = c(7, 4, 2), up = c(7, 5, 3),
            type = "standardized")
}
##########################################################################
##########################################################################
## defining function 'locally'
locally_det<- function(param,  auxiliary){
 ## param is the vector of theta = (theta0, theta1, theta2)
 ux <- 0
 lx <- 150
 xstar <- (ux + param[3]) * (lx + param[3]) * (log(ux + param[3]) -
  log(lx + param[3]))/(ux - lx) - param[3]
 denominator <- det(FIM_loglin(x = c(lx, xstar, ux) , w = rep(1/3, 3), param = param))
 return(denominator)
}
equivalence (fimfunc = "FIM_loglin",
            x = c(0, 4.2494, 17.0324, 149.9090),
            w = c(0.3204, 0.1207, 0.2293, 0.3296),
            lx = 0, ux = 150,
            lp = c(2, 2, 1), up = c(2, 2, 15),
            locally = locally_det,
            type = "standardized")
}
\references{
Atwood, C. L. (1969). Optimal and efficient designs of experiments. The Annals of Mathematical Statistics, 1570-1602.
}
\seealso{
\code{\link{print.equivalence}}, \code{\link{equivalence_ave}} and \code{\link{equivalence_multiple}}.
}

