#' @title Reduce Dimensionality
#' @description Reduce the dimensionality (i.e., the column number) of a design matrix to a desired level using Lasso.
#'
#' @param fit The fitted cross validation object generated by glmnet::cv.glmnet.
#' @param X An input design matrix whose column number is the dimensionality to be reduced.
#' @param bound The targeted number of dimensionality after reducing.
#'
#' @return A list of (1) index.X, indices of selected columns in the design matrix;
#'                   (2) sub.X, indices of selected columns in the design matrix.
#'
#' @export
#' @import glmnet
#'
#' @examples
#' set.seed(10)
#' X <- matrix(rnorm(100), nrow = 10)
#' Y <- matrix(rnorm(10), nrow = 10)
#' set.seed(11)
#' cvob1 <- glmnet::cv.glmnet(X, Y)
#' tmp <- reduce.dim(fit = cvob1, X = X, bound = 3)
reduce.dim <- function(fit, X, bound) {
  lambda.index <- max(which(fit$nzero <= bound))
  lambda_hat <- fit$lambda[lambda.index]
  beta.est <- stats::coef(fit, s = lambda_hat)
  selected.index <- beta.est@i[-1]
  X <- X[,selected.index]

  result <- list("index.X" = selected.index, "sub.X" = X)
  return (result)
}
