#' Plot Objects Generated by the Keyboard Design Package
#'
#' This function plots the objects returned by the other functions in the Keyboard package. \cr\cr
#'
#' (1) For get.oc.kb() and get.oc.comb.kb(), the operating characteristics of
#' the Keyboard design, which include: \cr
#' (i) the overall dose selection percentage of the simulations generated, \cr
#' (ii) the number of patients treated at each dose, and \cr
#' (iii) the toxicities presented at each dose. \cr\cr
#' (2) For get.oc.obd.kb() and get.oc.obd2.kb(), the operating characteristics of the Keyboard design for phase I/II, which includes: \cr
#' (i) the overall dose selection percentage of the simulations generated, \cr
#' (ii) the number of patients treated at each dose,  \cr
#' (iii) the toxicities presented at each dose, and \cr
#' (iv) the efficacies presented at each dose. \cr\cr
#' (3) For select.mtd.kb() and select.mtd.comb.kb(): an infographic showing the
#' estimates of the toxicity probability for each dose, with corresponding
#' 95\% credible intervals.
#' 
#' 
#'
#' @param x The object (returned by another function) to be plotted.
#' @param ... Ignored arguments.
#' @param name The name of the object to be plotted.\cr
#'             (Users should not input this parameter).
#'
#' @return This function returns a figure.
#' @author Hongying Sun, Li Tang, and Haitao Pan
#' @examples
#' \donttest{
#' ### Single-agent trial ###
#'
#' ## Get the operating characteristics for a Keyboard single-agent trial
#' oc <- get.oc.kb(target=0.3, p.true=c(0.05, 0.15, 0.3, 0.45, 0.6),
#'                 ncohort=10, cohortsize=3, ntrial=1000)
#' summary.kb(oc)
#'
#' plot.kb(oc)
#' plot.kb(oc$selpercent)
#' plot.kb(oc$npatients)
#' plot.kb(oc$ntox)
#'
#' ## Select the MTD based on a trial's data
#' n <- c(3, 3, 15, 9, 0)
#' y <- c(0, 0, 4, 4, 0)
#' selmtd <- select.mtd.kb(target=0.3, npts=n, ntox=y)
#' summary.kb(selmtd)
#'
#' plot.kb(selmtd)
#'
#'
#' ### Drug-combination trial ###
#'
#' ## Get the operating characteristics for a Keyboard drug-combination trial
#' p.true <- matrix(c(0.01, 0.03, 0.10, 0.20, 0.30,
#'                    0.03, 0.05, 0.15, 0.30, 0.60,
#'                    0.08, 0.10, 0.30, 0.60, 0.75), byrow=TRUE, ncol=5)
#' oc.comb <- get.oc.comb.kb(target=0.3, p.true=p.true, ncohort=20, 
#'  cohortsize=3, n.earlystop=12, startdose=c(1, 1), ntrial=100)
#' summary.kb(oc.comb)
#'
#' plot.kb(oc.comb)
#' plot.kb(oc.comb$selpercent)
#' plot.kb(oc.comb$nptsdose)
#' plot.kb(oc.comb$ntoxdose)
#'
#' ## Select the MTD based on a trial's data
#' n <- matrix(c(3, 5, 0, 0, 0,
#'               7, 6, 15, 0, 0,
#'               0, 0, 4, 0, 0), ncol=5, byrow=TRUE)
#' y <- matrix(c(0, 1, 0, 0, 0,
#'               1, 1, 4, 0, 0,
#'               0, 0, 2, 0, 0), ncol=5, byrow=TRUE)
#' sel.comb <- select.mtd.comb.kb(target=0.3, npts=n, ntox=y)
#' summary.kb(sel.comb)
#'
#' plot.kb(sel.comb)
#'
#' ### oc.obd.kb
#' toxicity.low <- 0.15
#' toxicity.moderate <- 0.33
#' toxicity.high <- 0.40
#' efficacy.low <- 0.20
#' efficacy.moderate <- 0.40
#' efficacy.high <- 0.60
#' target.toxicity <- 0.30
#' target.efficacy <- 0.40
#' p.true <-c(0.08,0.30,0.60,0.80)
#' q.true <- c(0.25,0.40,0.25,0.50)
#' oc.obd.kb <- get.oc.obd.kb(toxicity.low = toxicity.low,
#'              toxicity.moderate= toxicity.moderate,
#'              toxicity.high = toxicity.high,
#'              efficacy.low = efficacy.low,
#'              efficacy.moderate = efficacy.moderate,
#'              efficacy.high = efficacy.high,
#'              target.toxicity=target.toxicity,
#'              target.efficacy= target.efficacy,
#'              p.true= p.true, q.true= q.true)
#' summary.kb(oc.obd.kb)
#' plot.kb(oc.obd.kb)
#' plot.kb(oc.obd.kb$selpercent1)
#' plot.kb(oc.obd.kb$selpercent2)
#' plot.kb(oc.obd.kb$selpercent3)
#' plot.kb(oc.obd.kb$npatients)
#' plot.kb(oc.obd.kb$ntox)
#' plot.kb(oc.obd.kb$neff)
#' }
#' @seealso \code{\link{summary.kb}}
#' @export
plot.kb <- function(x, ..., name = deparse(substitute(x))) {
    # UseMethod("plot.kb")
    new.obj = unlist(strsplit(name, split = "\\$"))
    strpattern = "none"
    assign("objectPlot", get(new.obj[1])) #as.name() will convert a string into a reference
    if (length(new.obj) >= 2) {
        strpattern = new.obj[2]
    }
    ## If invalid option, e.g. plot.kb(oc.comb$flowchart)
    if (!is.element(strpattern, c("none", names(objectPlot)))) {
        stop("Please double check and specify the variable to be plotted...\n")
        ## edit: changed from warning()
    }

    ## plotting for get.oc.kb()
    if (!is.null(objectPlot$percentstop)) {
        # dev.new(); op <- par(no.readonly = TRUE); dev.off(); par(op)
        opar <- par(no.readonly = TRUE)
        on.exit(par(opar))
        par(mfrow = c(1,1), mar = c(5,5,4,2))

        if (length(new.obj) == 2) { ## WHEN TRUE??? Shouldn't it be length(x)?
            if (strpattern == "selpercent") {
                bplot = barplot(objectPlot$selpercent,
                                ylab = "selection percentage (%)", ylim = c(0,100),
                                cex.names = 1, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
            else if (strpattern == "npatients") {
                bplot = barplot(objectPlot$npatients,
                                ylab = "number of patients", ylim = c(0, max(objectPlot$npatients)),
                                cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$npatients)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
            else if (strpattern == "ntox") {
                bplot = barplot(objectPlot$ntox,
                                ylab = "number of toxicities", ylim = c(0, max(objectPlot$ntox)),
                                cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$ntox)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
        }
        else {
            opar <- par(no.readonly = TRUE)
            on.exit(par(opar))
            par(mfrow = c(2,1), mar = c(5,5,4,2))

            bplot = barplot(objectPlot$selpercent,
                            ylab = "selection percentage (%)", ylim = c(0,100),
                            cex.names = 1, xaxt = "n", cex.lab = 1.3)
            mtext("selection percentage", 3, line = 2, cex = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent)))
            mtext("dose level", 1, line = 2, cex = 1.3)

            bplot = barplot(objectPlot$npatients,
                            ylab = "number of patients", ylim = c(0, max(objectPlot$npatients)),
                            cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$npatients)))
            mtext("patient allocation", 3, line = 2, cex = 1.3)
            mtext("dose level", 1, line = 2, cex = 1.3)

            ## why not "number of toxicities" too?
        }
    }
    

    ## Plotting for get.oc.comb.kb()
    else if(!is.null(objectPlot$pcs)) { # | !is.null(objectPlot$pcs.contour)) {

        if (!requireNamespace("epade", quietly = TRUE)) {
                    stop("Package \"epade\" needed for this function to work.",
             "Please install it.", call. = FALSE)

        }
        # dev.new(); op <- par(no.readonly = TRUE); dev.off(); par(op)
        #par(mar=c(0,0,0,0),oma=c(0,0,0,0))
        opar <- par(no.readonly = TRUE)
        on.exit(par(opar))
        par(mfrow = c(1,1), mar = c(5,5,2,2))

        if (strpattern == "selpercent") {
            requireNamespace("epade", quietly = TRUE)
            epade::bar3d.ade(objectPlot$selpercent,
                             wall = 6, zw = 1, xw = 0.2,
                             yticks = seq(0,100,20),
                             zticks = seq(1:nrow(objectPlot$selpercent)),
                             xticks = seq(1:ncol(objectPlot$selpercent)),
                             xlab = "Drug B",
                             ylab = "selection percentage (%)",
                             zlab = "Drug A",
                             col = 'lavender', alpha = 0.5)

        }
        else if (strpattern == "nptsdose") { ## changed from BOIN's "npatients" (to match var in get.oc.comb.kb)
            requireNamespace("epade", quietly = TRUE)
            epade::bar3d.ade(objectPlot$nptsdose,
                             wall = 6, zw = 1, xw = 0.2,
                             yticks = round(seq(1,max(objectPlot$nptsdose),
                                            max(objectPlot$nptsdose)/7),0),
                             zticks = seq(1:nrow(objectPlot$nptsdose)),
                             xticks = seq(1:ncol(objectPlot$nptsdose)),
                             xlab = "Drug B",
                             ylab = "number of patients",
                             zlab = "Drug A",
                             col  = 'lavender', alpha = 0.5)

        }
        else if (strpattern == "ntoxdose"){ ## changed from BOIN's "ntox" (to match var in get.oc.comb.kb)
            requireNamespace("epade", quietly = TRUE)
            epade::bar3d.ade(objectPlot$ntoxdose,
                             wall = 6, zw = 1, xw = 0.2,
                             yticks = round(seq(1, max(objectPlot$ntoxdose),
                                            max(objectPlot$ntoxdose)/7),0),
                             zticks = seq(1:nrow(objectPlot$ntoxdose)),
                             xticks = seq(1:ncol(objectPlot$ntoxdose)),
                             xlab = "Drug B",
                             ylab = "number of toxicities",
                             zlab = "Drug A",
                             col = 'lavender', alpha = 0.5)

        }
        #
        #else if (strpattern=="flowchart"){
        #    if(is.null(objectPlot$pcs.contour)){# a single MTD
        #
        #        J=nrow(objectPlot$p.true);K=ncol(objectPlot$p.true)
        #        xlab='Drug B'; ylab='Drug A'
        #        if (J > K) { S=J; J=K; K=S; xlab='Drug A'; ylab='Drug B'}
        #
        #        xmax = K * 2 - 1
        #        xmin = 1
        #        ymax = 2*J
        #        ymin = 1
        #        ptcex = 1.5
        #
        #        par(mar=c(5,5,2,2))
        #        plot(1:xmax, xlim=c(xmin, xmax), ylim=c(ymin, ymax + 0.5),
        #             pch='',axes=F,xlab=xlab, ylab=ylab, cex.axis=1, cex.lab=1)
        #
        #        for(i in seq(1,xmax,by=2)) for(j in seq(1,ymax-1,by=2))
        #        points(i,j,pch=1,cex=ptcex)
        #
        #        if(J>2 & K>2){
        #
        #            arrows(3+0.1*J/K,3,3+1, 3,col=3,length=0.06,lty=1,lwd=2)
        #            arrows(3-0.1*J/K,3,3-1, 3,col=2,length=0.06,lty=1,lwd=2)
        #            arrows(3,3-0.1*J/K,3, 3-1,col=2,length=0.06,lty=1,lwd=2)
        #            arrows(3,3+0.1*J/K,3, 3+1,col=3,length=0.06,lty=1,lwd=2)
        #            points(3,3,pch=19,cex=ptcex)
        #        }
        #        if(J==2 | K==2){
        #            arrows(1+0.2*J/K,1,1+1, 1,col=3,length=0.06,lty=1,lwd=2)
        #            arrows(1,1+0.2*J/K,1, 1+1,col=3,length=0.06,lty=1,lwd=2)
        #            points(1,1,pch=19,cex=ptcex)
        #        }
        #
        #        #legend("top",y=ymax, c("escalation","de-escalation"),col = c(3, 2),lty=1,horiz=T,bty="n")#,inset=.05)
        #
        #        xx=quantile(1:xmax,c(1/4,3/4))
        #        text(x = xx[1]+0.1*ymax/(xmax+1), y = ymax, labels = "escalation", pos =4)
        #        arrows(xx[1]-1,ymax,xx[1], ymax,col=3,length=0.06,lty=1,lwd=2)
        #
        #        text(x = xx[2]+0.1*ymax/(xmax+1), y = ymax, labels = "de-escalation", pos = 4)
        #        arrows(xx[2]-1,ymax,xx[2], ymax,col=2,length=0.06,lty=1,lwd=2)
        #
        #
        #
        #    }
        #    else{# MTD contour
        #
        #        J=nrow(objectPlot$p.true);K=ncol(objectPlot$p.true)
        #        xlab='Drug B'; ylab='Drug A'
        #        if (J > K) { S=J; J=K; K=S; xlab='Drug A'; ylab='Drug B'}
        #
        #        xmax = K * 2
        #        xmin = - 0.8
        #        ymax = J*(J+1) + 2*J
        #        ymin = 1 - 0.5
        #        ptcex = 1.5
        #
        #        par(mar=c(5,5,4,2))
        #        plot(1:xmax, xlim=c(xmin, xmax), ylim=c(ymin, ymax + 0.5),
        #             pch='',axes=F,xlab=xlab, ylab=ylab,cex.axis=1, cex.lab=1)
        #
        #        active.rows = NULL
        #        for(j in 1:(J+1)) active.rows = c(active.rows, 1:J + (J+2)*(j-1))
        #
        #        for(i in seq(1,xmax, by=2)) for(j in 1:ymax)
        #        if(is.element(j, active.rows))
        #        points(i,j, pch=1, cex=ptcex)
        #
        #        text(-0.5, sort(seq(ymax-(J-1)/2, (J+1)/2, len=J+1),decreasing=FALSE),
        #             paste('(', letters[seq(J+1,1)], ')', sep='') , cex=1)
        #
        #        segments(0.5, ymax+0.5, xmax, ymax+0.5);
        #        segments(0.5, ymax-(J-1)-0.5, 0.5, ymax+0.5);
        #        segments(0.5, ymax-(J-1)-0.5, 1.5,ymax-(J-1)-0.5);
        #        segments(1.5, ymax-(J-1)-0.5, 1.5,ymax-0.5);
        #        segments(1.5, ymax-0.5, xmax, ymax-0.5);
        #        segments(xmax, ymax-0.5, xmax,ymax+0.5);
        #
        #        mtds = NULL; crows = NULL
        #        arrows(1,ymax-J+1+0.15*J/K,1,ymax-J+1+0.60,col=1,length=0.06,lty=1,lwd=2)
        #        tmpx = sort(sample(seq(1,xmax-2,2), J-1), decreasing=FALSE)
        #        for(j in 1:(J-1)){
        #            crow = ymax-J*j-2*j-j; crows = c(crows, crow)
        #            rect(2, crow-0.5, xmax, crow+0.5)
        #
        #            points(tmpx[j],crow+1, pch=8, cex=ptcex) # 2*j-1
        #            mtds = rbind(mtds, c(tmpx[j], J-j+1))  # 2*j-1
        #            if(tmpx[j]+2< xmax-1){
        #                arrows(tmpx[j]+2+0.2, crow, tmpx[j]+2+0.50, crow,col=1,length=0.08,lty=1,lwd=2)
        #            }else{ arrows(tmpx[j]+2-0.2, crow, tmpx[j]+2-0.5, crow,col=1,length=0.08,lty=1,lwd=2) }
        #        }
        #        mtds = rbind(mtds, c(xmax-1, 1))
        #        for(j in 1:J){
        #            x = mtds[j,1]; y = mtds[j,2]
        #            points(rep(x,y),y+(J+2)*0:(y-1),pch=8,cex=ptcex)
        #        }
        #    }
        #}
        #
        else {
            if (strpattern == "none") {
                # dev.new(); op <- par(no.readonly = TRUE); dev.off(); par(op)
                opar <- par(no.readonly = TRUE)
                on.exit(par(opar))

                par(mfrow = c(2,1), mar = c(4,3,2,2))
                requireNamespace("epade")
                epade::bar3d.ade(objectPlot$selpercent,
                                 wall = 6, zw = 1, xw = 0.2,
                                 main = "selection percentage",
                                 yticks = seq(0,100,20),
                                 zticks = seq(1:nrow(objectPlot$selpercent)),
                                 xticks = seq(1:ncol(objectPlot$selpercent)),
                                 xlab = "Drug B",
                                 ylab = "selection percentage (%)",
                                 zlab = "Drug A",
                                 col = 'lavender', alpha = 0.5)
                epade::bar3d.ade(objectPlot$nptsdose,
                                 wall = 6, zw = 1, xw = 0.2,
                                 main = "patient allocation",
                                 yticks = round(seq(1,max(objectPlot$nptsdose),
                                                max(objectPlot$nptsdose)/7),0),
                                 zticks = seq(1:nrow(objectPlot$nptsdose)),
                                 xticks = seq(1:ncol(objectPlot$nptsdose)),
                                 xlab = "Drug B",
                                 ylab = "number of patients",
                                 zlab = "Drug A",
                                 col = 'lavender', alpha = 0.5)

                ## why not "number of toxicities" too?
            }
            else {
                warning(paste("The variable (", strpattern,") to be plotted cannot be found in data object: ",
                        new.obj[1],".\n",sep=""))
            }
        }
    }

    ## Plotting for select.mtd.kb() or select.mtd.comb.kb()
    ## [the latter function only finds a single MTD, not a MTD contour]
    else if (!is.null(objectPlot$MTD)) {
        if (objectPlot$MTD[1] == 99) {
            stop("All tested doses are overly toxic. No MTD is selected!\n")
            ## edit: changed from warning()
        }
        else {
            if (!is.null(objectPlot$p_est)) {
                # dev.new(); op <- par(no.readonly = TRUE); dev.off(); par(op)
                opar <- par(no.readonly = TRUE)
                on.exit(par(opar))
                par(mfrow = c(1,1), mar = c(5,5,4,2))

                ## Note that this if-else is switched from that of the BOIN code.
                if (length(objectPlot$MTD) < 2) { ## select.mtd.kb()
                    p_est = objectPlot$p_est; p_hat = p_est[,2]
                    ci = p_est[,3]; ci = gsub("[\\(\\)]", "", ci)
                    conf.intv = matrix(unlist(strsplit(ci, ",")), byrow = TRUE, ncol = 2)
                    if (p_est[1,2] == "----") {
                        stop("The trial is stopped since the lowest dose is too toxic.\n")
                        ## edit: changed from warning()
                    }
                    else {
                        numbs = ifelse(sum(p_hat == "----") == 0,
                                       length(p_hat), min(which(p_hat == "----")) - 1)
                        numbs2 = length(p_hat)
                        phatx = as.numeric(as.character(p_hat[1:numbs]))
                        lwr = as.numeric(as.character(conf.intv[1:numbs,1]))
                        upr = as.numeric(as.character(conf.intv[1:numbs,2]))
                        opar <- par(no.readonly = TRUE)
                        on.exit(par(opar))
                        par(mar = c(5,5,4,2))
                        plot(1:numbs2, ylim = c(0,1), xlab = 'dose level',
                             ylab = 'DLT rate', pch = '', xaxt = "n", cex.lab = 1.3)
                        axis(1, at = 1:numbs2, labels = 1:numbs2)
                        abline(h = objectPlot$target, lty = 2, col = 2)
                        points(1:numbs, phatx, pch = 19)
                        arrows(x0 = 1:numbs, x1 = 1:numbs, y0 = lwr, y1 = upr,
                               code = 3, angle = 90, length = 0.1)
                        if (numbs < numbs2) {
                            points((numbs + 1):numbs2,
                                    seq(min(1, max(phatx, na.rm = T) + 0.05),
                                        min(max(phatx, na.rm = T) + 0.20, 1),
                                        length = numbs2 - numbs),
                                    pch = "*", cex = 1.5)
                            legend('topleft', '*   no patient treated')
                        }
                    }
                }
                else{ ## select.mtd.comb.kb()
                    if (!requireNamespace("epade", quietly = TRUE)) {
                                stop("Package \"epade\" needed for this function to work.",
             "Please install it.", call. = FALSE)

                    }

                    requireNamespace("epade", quietly = TRUE)
                    epade::bar3d.ade(objectPlot$p_est,
                                     wall = 6, zw = 1, xw = 0.2,
                                     yticks = round(seq(0,1,0.1),1),
                                     zticks = seq(1:nrow(objectPlot$p_est)),
                                     xticks = seq(1:ncol(objectPlot$p_est)),
                                     xlab = "Drug B",
                                     ylab = "DLT rate",
                                     zlab = "Drug A",
                                     col = 'lavender', alpha = 0.5)
                }
            }
        }
    }

        ## plotting for get.oc.obd.kb()
    else if (!is.null(objectPlot$name) && objectPlot$name == "get.oc.obd.kb") {
        # dev.new(); op <- par(no.readonly = TRUE); dev.off(); par(op)
        opar <- par(no.readonly = TRUE)
        on.exit(par(opar))
        par(mfrow = c(1,1), mar = c(5,5,4,2))

        if (length(new.obj) == 2) { ## WHEN TRUE??? Shouldn't it be length(x)?
            if (strpattern == "selpercent1") {
                bplot = barplot(objectPlot$selpercent1,
                                ylab = "selection percentage (%) using utility function 1", ylim = c(0,100),
                                cex.names = 1, xaxt = "n", cex.lab = 1.3)
                # mtext("selection percentage using utility function 1", 3, line = 2, cex = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent1)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }

            else if (strpattern == "selpercent2") {
                bplot = barplot(objectPlot$selpercent2,
                                ylab = "selection percentage (%) using utility function 2", ylim = c(0,100),
                                cex.names = 1, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent2)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
            else if (strpattern == "selpercent3") {
                bplot = barplot(objectPlot$selpercent3,
                                ylab = "selection percentage (%) using utility function 3", ylim = c(0,100),
                                cex.names = 1, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent3)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
            else if (strpattern == "npatients") {
                bplot = barplot(objectPlot$npatients,
                                ylab = "number of patients", ylim = c(0, max(objectPlot$npatients)),
                                cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$npatients)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
            else if (strpattern == "ntox") {
                bplot = barplot(objectPlot$ntox,
                                ylab = "number of toxicities", ylim = c(0, max(objectPlot$ntox)),
                                cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$ntox)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
            else if (strpattern == "neff") {
                bplot = barplot(objectPlot$neff,
                                ylab = "number of efficacies", ylim = c(0, max(objectPlot$ntox)),
                                cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
                axis(1, at = bplot, labels = seq(1, length(objectPlot$neff)))
                mtext("dose level", 1, line = 2, cex = 1.3)
            }
        }
        else {
            par(mfrow = c(3,1), mar = c(5,5,4,2))
            bplot = barplot(objectPlot$selpercent1,
                            ylab = "selection percentage (%) ", ylim = c(0,100),
                            cex.names = 1, xaxt = "n", cex.lab = 1.3)
            mtext("selection percentage using utility function 1", 3, line = 2, cex = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent1)))
            mtext("dose level", 1, line = 2, cex = 1.3)

            bplot = barplot(objectPlot$selpercent2,
                            ylab = "selection percentage (%) ", ylim = c(0,100),
                            cex.names = 1, xaxt = "n", cex.lab = 1.3)
            mtext("selection percentage using utility function 2", 3, line = 2, cex = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent2)))
            mtext("dose level", 1, line = 2, cex = 1.3)

            bplot = barplot(objectPlot$selpercent3,
                            ylab = "selection percentage (%) ", ylim = c(0,100),
                            cex.names = 1, xaxt = "n", cex.lab = 1.3)
            mtext("selection percentage using utility function 3", 3, line = 2, cex = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$selpercent3)))
            mtext("dose level", 1, line = 2, cex = 1.3)

            bplot = barplot(objectPlot$npatients,
                            ylab = "number of patients", ylim = c(0, max(objectPlot$npatients)),
                            cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$npatients)))
            mtext("patient allocation", 3, line = 2, cex = 1.3)
            mtext("dose level", 1, line = 2, cex = 1.3)

            ## number of toxicities
            bplot = barplot(objectPlot$ntox,
                            ylab = "number of toxicities", 
                            ylim = c(0, max(objectPlot$ntox)),
                            cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$ntox)))
            mtext("patient allocation for toxicities", 3, line = 2, cex = 1.3)
            mtext("dose level", 1, line = 2, cex = 1.3)

            bplot = barplot(objectPlot$neff,
                            ylab = "number of efficacies", 
                            ylim = c(0, max(objectPlot$neff)),
                            cex.names = 1, beside = FALSE, xaxt = "n", cex.lab = 1.3)
            axis(1, at = bplot, labels = seq(1, length(objectPlot$ntox)))
            mtext("patient allocation for efficacies", 3, line = 2, cex = 1.3)
            mtext("dose level", 1, line = 2, cex = 1.3)
        }
    }


    else {
        stop("Please double check and specify the variable to be plotted...\n")

    }
    par(opar)
}
