% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LA_opt.R
\name{LA_opt}
\alias{LA_opt}
\title{Lioness Algorithm for experimental designs with continuous input and function optimization}
\usage{
LA_opt(
  of = NULL,
  n = NULL,
  lb,
  ub,
  m = 10,
  N = 10,
  OC = NULL,
  type = "mini",
  alpha = 0.1,
  maxtime = 5
)
}
\arguments{
\item{of}{An objective function to be evaluated. If user is seeking for optimal design rather than function optimization, \code{of} should be left with \code{NULL}, which is the default setting.}

\item{n}{A positive integer, which stands for the number of rows (or run size) for a design. If user is seeking for optimal design, \code{n} should be the number of rows of the design matrix. If user is seeking for function optimization, \code{n} should be left with \code{NULL}, which is the default setting.}

\item{lb}{A vector contains the lower bounds of all the input variables. For example, if there are 3 input variables whose lower bounds are 0, 5, and 15, \code{lb} should be \code{lb=c(0,5,15)}.}

\item{ub}{A vector contains the upper bounds of all the input variables. For example, if there are 3 input variables whose upper bounds are 10, 15, and 25, \code{ub} should be \code{lb=c(10,15,25)}.}

\item{m}{A positive integer, which stands for the number of starting lionesses agents. The default is set to be 10, and it is recommended to be no greater than 100.}

\item{N}{A positive integer, which stands for the number of iterations. The default is set to be 10. A large value of \code{N} will result a high CPU time, and it is recommended to be no greater than 500.}

\item{OC}{An optimality criterion. If user is seeking for optimal design, \code{OC} should be an optimality criterion for how to evaluate the design matrix. If user is seeking for function optimization, \code{OC} should be left with \code{NULL}, which is the default setting.}

\item{type}{A logic input argument, which indicates the type of optimization. If \code{type} is \code{mini} (the default setting), minimization will be implemented in the algorithm. If \code{type} is \code{maxi},  maximization will be implemented in the algorithm.}

\item{alpha}{A tuning parameter in algorithm for controlling how big the change would be when updating elements in the step of avoiding local optimum. The default is set to be 0.1, which is the recommended value.}

\item{maxtime}{A positive number, which indicates the expected maximum CPU time given by user, and it is measured by minutes. For example, maxtime=3.5 indicates the CPU time will be no greater than three and half minutes. The default is set to be 5.}
}
\value{
If all inputs are logical, then the output will be either a \code{n} by \code{length(lb)} optimal design or a 1 by \code{length(lb)} vector of optimal solutions for function optimization. Here, the \code{length(lb)} is assumed to be at least 2 for either case.
}
\description{
\code{LA_opt} returns optimal designs with continuous input or optimal solutions for function optimization
}
\examples{
#We start with function optimization
#Now define an objective function: Sum of Different Powers
SDP=function(x){i=1:length(x);y=sum(abs(x)^(i=1));return(y)}

#Use LA to find the optimal solution under 20-dimensional setting
#for SDP function with 10 iterations.
try=LA_opt(of=SDP,lb=rep(-1,20),ub=rep(1,20),m=10,N=10,type="mini")
SDP(try)  #Note that the true global optimum is 0, but we only have 10 iterations

#Another example
#Define an objective function: Cross-in-Tray
CiT=function(x){x1=x[1];x2=x[2];y=-0.0001*(abs(sin(x1)*sin(x2)*
exp(abs(100-sqrt(x1^2+x2^2)/pi)))+1)^0.1;return(y)}

#Use LA to find the optimal solution under 2-dimensional setting
#for CiT function with 10 iterations.
try2=LA_opt(of=CiT,lb=rep(-10,2),ub=rep(10,2),m=10,N=10,type="mini")
CiT(try2) #Note that the true global optimum is -2.06261, but we only have 10 iterations

#Next we introduce the optimal design part
#Assume in a simple linear regression model, we want to find a D-optimal
#20-run design, where the input variable takes values between 0 and 24.
#In theory, we know the optimal design is the following:
#matrix(c(rep(1,20),rep(0,10),rep(24,10)),ncol=2,nrow=20,byrow=FALSE)
#Let's see if LA is able to identify that.

#Define the D-optimality criterion in simple linear regression model:
D=function(x){IM=t(x)\%*\%x;return(det(IM))}

#Use LA to find the optimal solution for above problem.
#We want to maximize the determinant of the information matrix.
try3=LA_opt(n=20,lb=c(1,0),ub=c(1,24),m=10,N=10,OC=D,type="maxi")
try3 #with more iterations, LA would return even better result.
}
