#' Lioness Algorithm for Latin hypercube design
#' 
#' \code{LA_LHDC} returns an optimal \code{n} by \code{k} Latin hypercube design (LHD) matrix generated by lioness algorithm (LA)
#'
#' @param n A positive integer that stands for the number of rows (or run size).
#' @param k A positive integer that stands for the number of columns (or factor size).
#' @param m A positive integer that stands for the number of starting design candidates. The default is set to be 100.
#' @param N A positive integer that stands for the maximum number of iterations. The default is set to be 5000. A larger value of \code{N} may result a high CPU time.
#' @param OC An optimality criterion. The default setting is "phi_p" (for space-filling LHDs), and it could be one of the following: "phi_p", "AvgAbsCor", "MaxAbsCor", "MaxProCriterion".
#' @param p A positive integer that is the parameter in the phi_p formula, and \code{p} is preferred to be large. The default and recommended value is 15.
#' @param q The default is set to be 1, and it could be either 1 or 2. If \code{q} is 1, the Manhattan (rectangular) distance will be used. If \code{q} is 2, the Euclidean distance will be used. 
#' 
#' @return If all inputs are logical, then the output will be an optimal \code{n} by \code{k} LHD.
#'
#' @examples
#' #generate a 6 by 3 maximin distance LHD with the default setting
#' try=LA_LHDC(n=6,k=3)
#' try
#'
#' #Another example
#' #generate a 9 by 4 nearly orthogonal LHD
#' try2=LA_LHDC(n=9,k=4,OC="AvgAbsCor")
#' try2
#' 
#' @export

LA_LHDC <- function(n, k, m = 100L, N = 5000L, OC = "phi_p", p = 15L, q = 1L) {
  .Call(`_LA_LA_LHDC`, n, k, m, N, OC, p, q)
}

#' @rdname LA_LHDC
#' 
#' @param x is a vector to be permuted.
#' 
permuC <- function(x) {
  .Call(`_LA_permuC`, x)
}

#' @rdname LA_LHDC
#' 
#' @param a is the starting value of the sequence.
#' @param b is the ending value of the sequence.
#' 
seqC <- function(a, b) {
  .Call(`_LA_seqC`, a, b)
}

#' @rdname LA_LHDC
#' 
#' @param n A positive integer that stands for the number of rows (or run size).
#' @param k A positive integer that stands for the number of columns (or factor size).
#' 
rLHDC <- function(n, k) {
  .Call(`_LA_rLHDC`, n, k)
}

#' @rdname LA_LHDC
#' 
#' @param X A matrix object. In general, \code{X} stands for the design matrix.
#' @param i A positive integer, which stands for the i^{th} row of \code{X}.
#' @param j A positive integer, which stands for the j^{th} row of \code{X}. Both \code{i} and \code{j} should be in [1,nrow(X)] and they should not be equal to each other.
#' @param q The default is set to be 1, and it could be either 1 or 2. If \code{q} is 1, the Manhattan (rectangular) distance will be used. If \code{q} is 2, the Euclidean distance will be used. 
#' 
dijC <- function(X, i, j, q = 1L) {
  .Call(`_LA_dijC`, X, i, j, q)
}

#' @rdname LA_LHDC
#' 
#' @param X A matrix object. In general, \code{X} stands for the design matrix.
#' @param p A positive integer that is the parameter in the phi_p formula, and \code{p} is preferred to be large. The default and recommended value is 15.
#' @param q The default is set to be 1, and it could be either 1 or 2. If \code{q} is 1, the Manhattan (rectangular) distance will be used. If \code{q} is 2, the Euclidean distance will be used. 
#'  
phi_pC <- function(X, p = 15L, q = 1L) {
  .Call(`_LA_phi_pC`, X, p, q)
}

#' @rdname LA_LHDC
#' 
#' @param X A matrix object. In general, \code{X} stands for the design matrix.
#' 
MaxProCriterionC <- function(X) {
  .Call(`_LA_MaxProCriterionC`, X)
}

#' @rdname LA_LHDC
#' 
#' @param x A vector.
#' @param y A vector.
#' 
corC <- function(x, y) {
  .Call(`_LA_corC`, x, y)
}

#' @rdname LA_LHDC
#' 
#' @param X A matrix object. In general, \code{X} stands for the design matrix.
#'
MaxAbsCorC <- function(X) {
  .Call(`_LA_MaxAbsCorC`, X)
}

#' @rdname LA_LHDC
#' 
#' @param X A matrix object. In general, \code{X} stands for the design matrix.
#'
AvgAbsCorC <- function(X) {
  .Call(`_LA_AvgAbsCorC`, X)
}

#' @rdname LA_LHDC
#' 
#' @param X A matrix object. In general, \code{X} stands for a design matrix.
#' @param j A positive integer, which stands for the j^{th} column (or row) of \code{X}, and it should be within [1,ncol(X)] (or [1,nrow(X)]).
#' @param type An exchange type. If \code{type} is "col" (the default setting), two random elements will be exchanged within column \code{j}. If \code{type} is "row", two random elements will be exchanged within row \code{j}.
#' 
exchangeC <- function(X, j, type = "col") {
  .Call(`_LA_exchangeC`, X, j, type)
}

