#' Orthogonal Latin Hypercube Design
#'
#' \code{OLHD2009} returns an orthogonal Latin hypercube design generated by the construction method of Lin et al. (2009)
#'
#' @param OLHD An orthogonal Latin hypercube design with run size \code{n} and factor size \code{p}, and it will be coupled with the input orthogonal array.
#' @param OA An orthogonal array, with \code{n^2} rows, \code{2f} columns, \code{n} symbols, strength two and index unity is available, which can be denoted as \code{OA(n^2,2f,n,2)}.
#'
#' @return If all inputs are logical, e,g. a \code{n} by \code{p} orthogonal Latin hypercube design and an \code{OA(n^2,2f,n,2)} orthogonal array, then the output will be an orthogonal Latin hypercube design with the following run size: \code{n^2} and the following factor size: \code{2fp}.
#'
#' @references Lin, C.D., Mukerjee, R., and Tang, B. (2009) Construction of orthogonal and nearly orthogonal Latin hypercubes. \emph{Biometrika}, \strong{96}(1), 243-247.
#'
#' @examples
#' #create a 5 by 2 OLHD
#' OLHD=OLHD2007(m=2)
#'
#' #create an OA(25,6,5,2)
#' OA=matrix(c(2,2,2,2,2,1,2,1,5,4,3,5,3,2,1,5,4,5,1,5,4,3,2,5,
#' 4,1,3,5,2,3,1,2,3,4,5,2,1,3,5,2,4,3,1,1,1,1,1,1,4,3,2,1,5,5,
#' 5,5,5,5,5,1,4,4,4,4,4,1,3,1,4,2,5,4,3,3,3,3,3,1,3,5,2,4,1,3,
#' 3,4,5,1,2,2,5,4,3,2,1,5,2,3,4,5,1,2,2,5,3,1,4,4,1,4,2,5,3,4,
#' 4,2,5,3,1,4,2,4,1,3,5,3,5,3,1,4,2,4,5,2,4,1,3,3,5,1,2,3,4,2,
#' 4,5,1,2,3,2),ncol=6,nrow=25,byrow=TRUE)
#'
#' #Construct a 25 by 12 OLHD
#' OLHD2009(OLHD,OA)
#'
#' @export

OLHD2009=function(OLHD,OA){
  n1=dim(OLHD)[1]
  k=dim(OLHD)[2]

  n2=length(unique(OA[,1]))
  f=dim(OA)[2]/2

  if(n1!=n2){
    stop("The number of rows of input OLHD must be equal to the number of levels of input orthogonal array")
  }

  if(dim(OA)[2]%%2!=0){
    stop("The number of columns of input orthogonal array must be an even number")
  }

  if(sqrt(dim(OA)[1])!=length(unique(OA[,1]))){
    stop("The number of rows of input orthogonal array taken square root must be equal to the number of levels of input orthogonal array")
  }

  A=array(OA,dim=c(n2^2,2*f,k))
  M=array(0,dim=c(n2^2,2*f,k))

  V=matrix(c(1,-n2,n2,1),ncol=2,nrow=2,byrow=T)

  X=NULL

  for (i in 1:k) {

    for (j in 1:n2) {      #step 1 starts

      for (m in 1:(2*f)) {
        location=which(A[,m,i]==j)
        A[location,m,i]=OLHD[j,i]
      }

    }                     #step 1 ends
  }


  for (i in 1:k) {

    for (j in 1:f) {   #step 2 starts

      M[,(2*j-1):(2*j),i]=A[,(2*j-1):(2*j),i]%*%V

    }                 #step 2 ends


    X=cbind(X,M[,,i])   #step 3

  }

  X
}
