% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Zagent.R
\name{Agent}
\alias{Agent}
\title{Agent Class for LLM Interactions}
\description{
An R6 class representing an agent that interacts with language models.
Each Agent can maintain its own memory, knowledge, and references to a model config.
}
\examples{
\dontrun{
# Example: Multi-Agent Conversation on Carbon Footprint
# ------------------------------------------------------
library(LLMR)

# Set up model configuration
agentcfg <- llm_config(
  provider   = "openai",
  model      = "gpt-4o-mini",
  api_key    = Sys.getenv("OPENAI_KEY"),
  temperature = 1.0,
  max_tokens  = 1000
)

# Create three agents: liberal, conservative, mediator
liberal_agent <- Agent$new(
  id = "liberal",
  model_config = agentcfg,
  knowledge = list(
    ideology  = "liberal",
    verbosity = "brief"
  )
)

conservative_agent <- Agent$new(
  id = "conservative",
  model_config = agentcfg,
  knowledge = list(
    ideology  = "conservative",
    verbosity = "very terse"
  )
)

mediator_agent <- Agent$new(
  id = "mediator",
  model_config = agentcfg,
  knowledge = list(role = "mediator")
)

# We can inject messages to establish context
# conv$add_message(
#   "System",
#   "We are discussing the most effective ways to reduce carbon footprint."
# )
# conv$add_message(
#   "Liberal",
#   "If corporations are not held accountable, they will continue to pollute.
#    We need stricter regulations."
# )
# conv$add_message(
#   "Conservative",
#   "Regulations can stifle economic growth. We should incentivize companies
#    to reduce emissions voluntarily. Don't kill capitalism!"
# )

# Create a conversation about reducing carbon footprint
# Chain agent responses:
conv <- LLMConversation$new(topic = "Reducing Carbon Footprint Discussion") +
  AgentAction(
    liberal_agent,
    prompt_template = "{{topic}}\n From your liberal perspective, what
    strategies would you suggest to further reduce the carbon footprint?"
  )

# See how conversation is injected into the prompt:
conv <- conv + AgentAction(
  conservative_agent,
  prompt_template = "so far we heard\n {{conversation}}\n Do you agree
  with all of this?  Given your coservative background, give your comments
  about every item proposed so far and then add your own suggestions."
)

conv <- conv + AgentAction(
  mediator_agent,
  prompt_template = "Considering these views\n{{conversation}}\n identify
  common ground and propose a unified approach to reducing the carbon
  footprint."
)

# New step for the liberal agent that now includes just the last line
conv <- conv + AgentAction(
  liberal_agent,
  prompt_template = "Reviewing what the moderator summarized: {{last_output}}\n
  I understand there is disagreement, but do you agree with these common grounds?
  If not, what would you change?"
)

conv$print_history()

# Another technique for more direct control of what goes into a prompt
mediator_snippet <- paste(
  tail(conv$conversation_history, 2)[[1]]$speaker,
  tail(conv$conversation_history, 2)[[1]]$text,
  collapse = ":"
)

conv <- conv + AgentAction(
  conservative_agent,
  prompt_template = "The mediator said:\n{{frozen_text}}\nWhat is your response?",
  replacements = list(frozen_text = mediator_snippet)
)

# Print the final conversation history
conv$print_history()
}

}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{id}}{Unique ID for this Agent.}

\item{\code{context_length}}{Maximum number of conversation turns stored in memory.}

\item{\code{model_config}}{The \code{llm_config} specifying which LLM to call.}

\item{\code{memory}}{A list of speaker/text pairs that the agent has "memorized."}

\item{\code{knowledge}}{Named list for additional agent-specific info/attributes.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Agent-new}{\code{Agent$new()}}
\item \href{#method-Agent-add_memory}{\code{Agent$add_memory()}}
\item \href{#method-Agent-generate_prompt}{\code{Agent$generate_prompt()}}
\item \href{#method-Agent-call_llm_agent}{\code{Agent$call_llm_agent()}}
\item \href{#method-Agent-generate}{\code{Agent$generate()}}
\item \href{#method-Agent-think}{\code{Agent$think()}}
\item \href{#method-Agent-respond}{\code{Agent$respond()}}
\item \href{#method-Agent-reset_memory}{\code{Agent$reset_memory()}}
\item \href{#method-Agent-clone}{\code{Agent$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-new"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-new}{}}}
\subsection{Method \code{new()}}{
Create a new Agent instance.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$new(id, context_length = 5, knowledge = NULL, model_config)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{Character. The agent's unique identifier.}

\item{\code{context_length}}{Numeric. The maximum memory length (default 5).}

\item{\code{knowledge}}{A named list of knowledge or persona details.}

\item{\code{model_config}}{An \code{llm_config} object specifying LLM settings.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-add_memory"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-add_memory}{}}}
\subsection{Method \code{add_memory()}}{
Add a new message to the agent's memory.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$add_memory(speaker, text)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{speaker}}{Character. The speaker name or ID.}

\item{\code{text}}{Character. The message content.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-generate_prompt"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-generate_prompt}{}}}
\subsection{Method \code{generate_prompt()}}{
Replace placeholders in a prompt template with values from `replacements`.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$generate_prompt(template, replacements = list())}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{template}}{Character. The prompt template.}

\item{\code{replacements}}{A named list of placeholder values.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Character. The prompt with placeholders replaced.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-call_llm_agent"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-call_llm_agent}{}}}
\subsection{Method \code{call_llm_agent()}}{
Call the underlying LLM with a plain text `prompt`.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$call_llm_agent(prompt, verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{prompt}}{Character. The final prompt to send.}

\item{\code{verbose}}{Logical. If TRUE, prints verbose info. Default FALSE.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list with $text (the LLM response) plus token usage, etc.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-generate"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-generate}{}}}
\subsection{Method \code{generate()}}{
Generate an LLM response using a prompt template and optional replacements.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$generate(prompt_template, replacements = list(), verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{prompt_template}}{Character. The prompt template.}

\item{\code{replacements}}{Named list for placeholder substitution in the prompt.}

\item{\code{verbose}}{Logical. If TRUE, prints extra info. Default FALSE.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list with $text, $tokens_sent, $tokens_received, and $full_response.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-think"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-think}{}}}
\subsection{Method \code{think()}}{
Have the agent produce an "internal" thought about a topic, using memory, etc.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$think(topic, prompt_template, replacements = list(), verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{topic}}{Character. A short description or label for the thought.}

\item{\code{prompt_template}}{Character. The prompt template.}

\item{\code{replacements}}{Named list. Additional placeholders to fill.}

\item{\code{verbose}}{Logical. If TRUE, prints extra info. Default FALSE.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-respond"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-respond}{}}}
\subsection{Method \code{respond()}}{
Have the agent produce a "public" answer or response about a topic.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$respond(topic, prompt_template, replacements = list(), verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{topic}}{Character. A short label for the question or request.}

\item{\code{prompt_template}}{Character. The prompt template.}

\item{\code{replacements}}{Named list. Placeholders to fill in the prompt.}

\item{\code{verbose}}{Logical. If TRUE, prints extra info. Default FALSE.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-reset_memory"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-reset_memory}{}}}
\subsection{Method \code{reset_memory()}}{
Reset the agent's memory (clear any stored conversation context).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$reset_memory()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Agent-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Agent-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Agent$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
