\name{monoisotopicPeaks-methods}
\docType{methods}
\alias{monoisotopicPeaks}
\alias{monoisotopicPeaks,MassPeaks-method}
\alias{monoisotopicPeaks,list-method}
\title{Finds monoisotopic peaks in a MassPeaks object.}
\description{
This method looks for monoisotopic peaks in peak list data
(represented by a \code{\linkS4class{MassPeaks}} objects).\cr
It is based on the poisson model for isotopic patterns described in Breen et al
2000.
}
\usage{
\S4method{monoisotopicPeaks}{MassPeaks}(object,
  minCor=0.95, tolerance=1e-4, distance=1.00235, size=3L:10L)
\S4method{monoisotopicPeaks}{list}(object, \dots)
}
\arguments{
  \item{object}{\code{\linkS4class{MassPeaks}} object or a \code{list} of
    \code{\linkS4class{MassPeaks}} objects.}
  \item{minCor}{\code{double}, minimal correlation between the peak pattern
    generated by the model and the experimental peaks in the
    \code{\linkS4class{MassPeaks}} object to be recognized as isotopic pattern.}
    \item{tolerance}{\code{double}, maximal relative deviation of peaks
      position (mass) to be considered as isotopic distance (
      \code{abs(((mass[i]+distance)-mass[i+1])/mass[i]) < tolerance}).}
    \item{distance}{\code{double}, distance between two consecutive peaks in an
      isotopic pattern (default value taken from Park et al 2008). Could contain
      more than one value, e.g. \code{distance=(1:3)^-1} to find isotopic
      patterns for multiple charged patterns (e.g. 1+, 2+, and 3+). Please note
      that the order matters here if there is a monoisotopic peak for charge
      state 1 and 3 it would be reported as charge 1 for
      \code{distance=(1:3)^-1} and as 3 for \code{distance=(3:1)^-1}
      respectively.}
  \item{size}{\code{double}, size (length) of isotopic pattern, longer patterns
      are prefered over shorter ones.}
  \item{\dots}{arguments to be passed to
    \code{\link[MALDIquant]{monoisotopicPeaks,MassPeaks-method}}. If
    \code{object} is a \code{list} \code{mc.cores} is also supported.}
}
\value{
Returns a \code{\linkS4class{MassPeaks}} object with monoisotopic peaks only.
}
\author{
Sebastian Gibb \email{mail@sebastiangibb.de}
}
\references{
K. Park, J.Y. Yoon, S. Lee, E. Paek, H. Park, H.J. Jung, and S.W. Lee. 2008.
Isotopic peak intensity ratio based algorithm for determination of isotopic
clusters and monoisotopic masses of polypeptides from high-resolution
mass spectrometric data.
Analytical Chemistry, 80: 7294-7303.

E.J. Breen, F.G. Hopwood, K.L. Williams, and M.R. Wilkins. 2000.
Automatic poisson peak harvesting for high throughput protein identification.
Electrophoresis 21: 2243-2251.
}
\seealso{
\code{\linkS4class{MassPeaks}},
\code{\link[MALDIquant]{detectPeaks,MassSpectrum-method}}

Website: \url{https://www.strimmerlab.org/software/maldiquant/}
}
\examples{
## load package
library("MALDIquant")

## create example peaks
p <- createMassPeaks(mass=995:1005,
                     intensity=c(100, 10, 30, 10, 40, # noise
                                 550, 330, 110, 10,   # isotopic pattern
                                 5, 15))              # more noise
m <- monoisotopicPeaks(p)
as.matrix(m)

## plot the peaks and mark the monoisotopic one
plot(p)
points(m, col=2, pch=4)
}
\keyword{methods}
