% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mbnma-class.R
\name{get.relative}
\alias{get.relative}
\title{Calculates relative effects/mean differences at a particular time-point}
\usage{
get.relative(
  mbnma,
  mbnma.add = NULL,
  time = max(mbnma$model.arg$jagsdata$time, na.rm = TRUE),
  treats = unique(c(mbnma$network$treatments, mbnma.add$network$treatments)),
  classes = NULL,
  lim = "cred"
)
}
\arguments{
\item{mbnma}{An S3 object of class \code{"mbnma"} generated by running
a time-course MBNMA model}

\item{mbnma.add}{An S3 object of \code{class("mbnma")} generated by running
a time-course MBNMA model. This should only be specified if results from two different MBNMA models
are to be combined to perform a 2-stage MBNMA (see Details).}

\item{time}{A numeric value for the time at which to estimate relative effects/mean differences.}

\item{treats}{A character vector of treatment names for which to calculate relative effects/mean
differences. Must be a subset of \code{mbnma$network$treatments}.}

\item{classes}{A character vector of class names for which to calculate relative effects/mean
differences. Must be a subset of \code{mbnma$network$classes}. Only works for class effect models.}

\item{lim}{Specifies calculation of either 95\% credible intervals (\code{lim="cred"}) or 95\% prediction intervals (\code{lim="pred"}).}
}
\value{
An object of class \code{"relative.array"} list containing:
\itemize{
\item The time-point for which results are estimated
\item Matrices of posterior means, medians, SDs and upper and lower 95\% credible intervals for the
differences between each treatment
\item An array containing MCMC results for the differences between all treatments specified in \code{treats}
or all classes specified in \code{classes}.
}

Results are reported in tables as the row-defined treatment minus the column-defined treatment.
}
\description{
Uses mbnma time-course parameter estimates to calculate treatment
differences between treatments or classes at a particular time-point.
Can be used to compare treatments evaluated in studies at different follow-up times, or even
to compare treatments in different MBNMA models via a common comparator.
}
\details{
\code{get.relative()} can also be used to perform a 2-stage MBNMA that allows synthesis of results
from two different MBNMA models via a single common comparator.
In an MBNMA model, all treatments must share the same time-course function. However, a 2-stage
approach can enable fitting of different time-course functions to different sets ("subnetworks") of
treatments. For example, some treatments may have rich time-course information,
allowing for a more complex time-course function to be used, whereas others may be sparse,
requiring a simpler time-course function.

Relative comparisons between treatments in the two datasets at specific follow-up times
can then be estimated from MBNMA predicted effects versus a common comparator
using the Bucher method and assuming consistency. See the MBNMAtime vignette for further details.
}
\examples{
\donttest{
# Create an mb.network object from a dataset
alognet <- mb.network(alog_pcfb)

# Run a quadratic time-course MBNMA using the alogliptin dataset
mbnma <- mb.run(alognet,
  fun=tpoly(degree=2,
  pool.1="rel", method.1="random",
  pool.2="rel", method.2="common"
  )
)

# Calculate differences between all treatments at 20 weeks follow-up
allres <- get.relative(mbnma, time=20)

# Calculate difference between a subset of treatments at 10 weeks follow-up
subres <- get.relative(mbnma, time=10,
  treats=c("alog_50", "alog_25", "placebo"))



###########################
##### 2-stage MBNMA #####
###########################

# Using the osteoarthritis dataset
# With placebo (Pl_0) as common comparator between subnetworks

#### Sparse model ####

# Treatments on which time-course data is limited
sparse.trt <- c("Ce_100", "Ce_400", "Du_90", "Lu_200", "Lu_400",
  "Lu_NA", "Et_5", "Ox_44")

# Create a subnetwork of studies comparing these treatments
sparse.df <- osteopain \%>\% dplyr::group_by(studyID) \%>\%
  dplyr::filter(any(treatment \%in\% sparse.trt)) \%>\%
  dplyr::ungroup() \%>\%
  subset(treatment \%in\% c("Pl_0", sparse.trt))

sparse.net <- mb.network(sparse.df)

# Run a ITP MBNMA with a known rate
sparse.mbnma <- mb.run(sparse.net, fun=titp(method.rate=0.8, pool.rate="abs"))


#### Complex model ####

# Treatments on which time-course data is rich
rich.trt <- levels(osteopain$treatment)[!levels(osteopain$treatment) \%in\%
  c("Pl_0", "Ce_100", "Ce_400", "Du_90", "Lu_200",
    "Lu_400", "Lu_NA", "Et_5", "Ox_44")]

# Create a subnetwork of studies comparing these treatments
rich.df <- osteopain \%>\% dplyr::group_by(studyID) \%>\%
  dplyr::filter(any(treatment \%in\% rich.trt)) \%>\%
  dplyr::ungroup() \%>\%
  subset(treatment \%in\% c("Pl_0", rich.trt))

rich.net <- mb.network(rich.df)

# Run a Emax MBNMA
rich.mbnma <- mb.run(rich.net, temax(p.expon = FALSE))


#### Calculate relative effects between models ####

# At 10 weeks follow-up
rels.sparse <- get.relative(sparse.mbnma, time=10)
rels.rich <- get.relative(rich.mbnma, time=10)

rels.all <- get.relative(mbnma=rich.mbnma,
  mbnma.add=sparse.mbnma, time=10)

print(rels.all$median)


}
}
