% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MSmix_functions_package.R
\name{var_spear_dist}
\alias{var_spear_dist}
\title{Variance of the Spearman distance}
\usage{
var_spear_dist(theta, n_items, log = TRUE)
}
\arguments{
\item{theta}{Non-negative precision parameter.}

\item{n_items}{Number of items.}

\item{log}{Logical: whether the expected Spearman distance on the log scale must be returned. Defaults to \code{TRUE}.}
}
\value{
Either the exact or the approximate (log-)variance of the Spearman distance under the Mallow model with Spearman distance.
}
\description{
Compute (either the exact or the approximate) (log-)variance of the Spearman distance under the Mallow model with Spearman distance.
}
\details{
When \eqn{n\leq 20}, the variance is exactly computed by relying on the Spearman distance distribution
provided by OEIS Foundation Inc. (2023). When \eqn{n>20}, it is approximated with the method introduced by Crispino et al. (2023) and, if \eqn{n>170}, the approximation is also restricted over a fixed grid of values for the Spearman distance to limit computational burden.

When \eqn{\theta=0}, this is equal to \eqn{\frac{n^2(n-1)(n+1)^2}{36}}, which is the variance of the Spearman
distance under the uniform (null) model.

The variance of the Spearman distance is independent of the consensus ranking of the Mallow model with Spearman distance due to the right-invariance
of the metric.
}
\examples{

## Example 1. Variance of the Spearman distance under the uniform (null) model,
## coinciding with n^2(n-1)(n+1)^2/36.
n_items <- 10
var_spear_dist(theta = 0, n_items= n_items, log = FALSE)
n_items^2*(n_items-1)*(n_items+1)^2/36

## Example 2. Variance of the Spearman distance.
var_spear_dist(theta = 0.5, n_items = 10, log = FALSE)

## Example 3. Log-variance of the Spearman distance as a function of theta.
var_spear_dist_vec <- Vectorize(var_spear_dist, vectorize.args = "theta")
curve(var_spear_dist_vec(x, n_items = 10),
  from = 0, to = 0.1, lwd = 2, col = 2,
  xlab = expression(theta), ylab = expression(log(V[theta](D))),
  main = "Log-variance of the Spearman distance")

## Example 4. Log--variance of the Spearman distance for varying number of items
# and values of the concentration parameter.
var_spear_dist_vec <- Vectorize(var_spear_dist, vectorize.args = "theta")
curve(var_spear_dist_vec(x, n_items = 10),
  from = 0, to = 0.1, lwd = 2, col = 2, ylim = c(5, 14),
  xlab = expression(theta), ylab = expression(log(V[theta](D))),
  main = "Log-variance of the Spearman distance")
curve(var_spear_dist_vec(x, n_items = 20), add = TRUE, col = 3, lwd = 2)
curve(var_spear_dist_vec(x, n_items = 30), add = TRUE, col = 4, lwd = 2)
legend("topright", legend = c(expression(n == 10), expression(n == 20), expression(n == 30)),
  col = 2:4, lwd = 2, bty = "n")

}
\references{
Crispino M., Mollica C., Astuti V. and Tardella L. (2023). Efficient and accurate inference for mixtures of Mallows models with Spearman distance. \emph{Statistics and Computing}, \bold{33}(98), DOI: 10.1007/s11222-023-10266-8.

OEIS Foundation Inc. (2023). The On-Line Encyclopedia of Integer Sequences, Published electronically at \url{https://oeis.org}

Kendall, M. G. (1970). Rank correlation methods. 4th ed. Griffin London.
}
