% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FRindex.R
\docType{class}
\name{FRindex}
\alias{FRindex}
\alias{FRindex-class}
\alias{FR_gam}
\alias{predict.FRindex}
\title{Functional Regression Indices & Weights}
\usage{
FRindex(formula, data, sign_prob = 0.5, ...)

FR_gam(
  formula,
  data,
  xarg = as.double(colnames(X)),
  family,
  knot_pct = 0.4,
  knot.value = ceiling(length(xarg) * knot_pct),
  ...
)

\method{predict}{FRindex}(
  object,
  newdata = object@data,
  newX = newdata[[object@formula[[3L]]]],
  new_xarg = as.double(colnames(newX)),
  ...
)
}
\arguments{
\item{formula}{a two-sided \link[stats]{formula}.
\describe{
\item{Left-hand-side}{is the \link[base]{name} of the response \eqn{y}.
Supported types of responses are \link[base]{double}, \link[base]{logical} and \link[survival]{Surv}.}
\item{Right-hand-side}{is the \link[base]{name}
of the tabulated \link[base]{double} \link[base]{matrix} \eqn{X} of functional predictor values.
Each row of \eqn{X} represents the tabulated values for a subject.
All rows/subjects are tabulated on a common grid \code{xarg}.
Each column of \eqn{X} represents the tabulated values at a point on the common grid for each subject.}
}}

\item{data}{\link[base]{data.frame}, with
the response \eqn{y} and the tabulated functional predictor values \eqn{X}
specified in \code{formula}.
If the functional predictor is the \link[stats]{quantile} function,
then \code{data} is preferably the returned object of \code{\link[=clusterQp]{clusterQp()}}.}

\item{sign_prob}{\link[base]{double} scalar between 0 and 1,
probability corresponding to the selected nearest-even quantile in \code{xarg},
which is used to define the \link[base]{sign} of the functional regression weights.
Default is \code{.5}, i.e., the nearest-even \link[stats]{median} of \code{xarg}}

\item{...}{for function \code{\link[=predict.FRindex]{predict.FRindex()}} and helper function \code{\link[=FR_gam]{FR_gam()}},
these are currently not in use.
For function \code{\link[=FRindex]{FRindex()}}, see a detailed explanation in section \strong{Using \code{...} in \code{FRindex()}}}

\item{xarg}{strictly increasing \link[base]{double} \link[base]{vector},
the common grid on which the functional predictor values \eqn{X} are tabulated}

\item{family}{\link[stats]{family} object, the distribution
and link function to be used in \link[mgcv]{gam}.
Default family for \link[survival]{Surv} response is \code{mgcv::cox.ph()},
for \link[base]{logical} response is \code{binomial(link = 'logit')},
for \link[base]{double} response is \code{gaussian(link = 'identity')}.}

\item{knot_pct}{positive \link[base]{double} scalar,
percentage of the number of columns of \eqn{X},
to be used as \code{knot.value}.
Default is \eqn{40\%}.
If \code{knot.value} is provided by the end-user, then \code{knot_pct} is ignored.}

\item{knot.value}{positive \link[base]{integer} scalar, number of knots
(i.e., parameter \code{k} in the spline smooth function \link[mgcv]{s})
used in \link[mgcv]{gam}.
Default is the \link[base]{ceiling} of \code{knot_pct} of
the column dimension of \eqn{X}}

\item{object}{an \linkS4class{FRindex} object for the \link[stats]{predict} method,
the returned object from function \code{\link[=FRindex]{FRindex()}}}

\item{newdata}{\link[base]{data.frame}, with at least
the tabulated functional predictor values \eqn{X^{new}}
based on \code{object@formula}}

\item{newX}{\link[base]{double} \link[base]{matrix},
functional predictor values \eqn{X^{new}} for a set of new subjects.
Each row of \eqn{X^{new}} represents the tabulated values for a new subject.
All rows/subjects are tabulated on a common grid \code{new_xarg}.
Each column of \eqn{X^{new}} represents the tabulated values at a point on the common grid for each new subject.}

\item{new_xarg}{strictly increasing \link[base]{double} \link[base]{vector},
the common grid on which the functional predictor values \eqn{X^{new}} are tabulated.
The length of \code{new_xarg}
does not need to be the same as the length of \code{object@xarg},
but they must share the same range.}
}
\value{
\subsection{Functional regression indices & weights model}{

Function \code{\link[=FRindex]{FRindex()}} returns an \link[base]{S4} \linkS4class{FRindex} object.
The slots of \link[base]{S4} class \linkS4class{FRindex} are described in section \strong{Slots}.
}

\subsection{Predict method for functional regression indices & weights}{

Function \code{\link[=predict.FRindex]{predict.FRindex()}} returns a
\link[base]{double} \link[base]{vector},
which is the predicted functional regression indices.
The returned object contains an \link[base]{attributes}
\describe{
\item{\code{attr(,'weight')}}{\link[base]{double} \link[base]{vector},
the predicted functional regression weights}
}
}
}
\description{
Functions explained in this documentation are,

\describe{

\item{\code{\link[=FRindex]{FRindex()}}}{
to compute the functional regression indices and weights based on the functional predictors.}

\item{\code{\link[=predict.FRindex]{predict.FRindex()}}}{
to compute the predicted values based on functional regression indices and weights model.}

\item{\code{\link[=FR_gam]{FR_gam()}}}{
a helper function to fit a functional regression model
using generalized additive models with integrated smoothness estimation (\link[mgcv]{gam}).}

}
}
\details{
\subsection{Functional regression indices & weights model}{

Function \code{\link[=FRindex]{FRindex()}} defines and calculates
the functional regression indices and weights in the following steps.

\enumerate{

\item Fit a functional regression model to the response \eqn{y}
using the functional predictor \eqn{X},
with tabulated tabulated on a same grid \code{xarg} for all subjects,
using helper function \code{\link[=FR_gam]{FR_gam()}}

\item Select one point in the tabulating grid \code{xarg}.
For one-dimensional domain,
we select the nearest-even \link[stats]{quantile} of the tabulating grid \code{xarg},
corresponding to the user-specified probability \code{sign_prob}.
Default \code{sign_prob = .5} indicates the \link[stats]{median} of \code{xarg}.

\item Obtain the fitted coefficient function \eqn{\hat\beta(x)},
tabulated on the grid \code{xarg},
using internal helper function \code{\link[=gam2beta]{gam2beta()}}

\item Calculate the integral of the product of
the fitted coefficient function \eqn{\hat\beta(x)} (from Step 3) and
the functional predictor values \eqn{X},
using the \link[pracma]{trapz}oid rule

\item Obtain the \link[base]{sign} of the \link[stats]{cor}relation between
\itemize{
\item the subject-specific functional predictor \emph{values},
at the selected quantile of \code{xarg} (from Step 2), and
\item the subject-specific integrals from Step 4
}

}

\emph{Functional regression weights} (slot \verb{@weight})
are the tabulated weight function on the grid \code{xarg}.
These weights are defined as the product of
\code{sign} (from Step 5) and \eqn{\hat\beta(x)} (from Step 3).

\emph{Functional regression indices} (slot \verb{@index})
are defined as the product of
\code{sign} (from Step 5) and \code{intg} (from Step 4).
Multiplication by \code{sign} is required to ensure
that the resulting functional regression indices
are positively associated with the functional predictor values
at the selected quantile of \code{xarg} (from Step 2).
}

\subsection{Predict method for functional regression indices & weights}{

Function \code{\link[=predict.FRindex]{predict.FRindex()}} computes functional regression indices and weights
based on the tabulated functional predictors \eqn{X^{new}} in a new sets of subjects.
It's important that the new tabulation grid \code{new_xarg} must have the same \link[base]{range}
as the model tabulation grid \code{object@xarg}.
Then,

\enumerate{

\item Obtain the fitted coefficient function \eqn{\hat\beta(x^{new})}
of the existing generalized additive model \code{object@gam},
but tabulated on the new grid \code{new_xarg},
using internal helper function \code{\link[=gam2beta]{gam2beta()}}

\item Calculate the integral of the product of
the fitted coefficient function \eqn{\hat\beta(x^{new})} (from Step 1) and
the new functional predictor values \eqn{X^{new}},
using the \link[pracma]{trapz}oid rule

}

Predicted functional regression weights
are the tabulated weight function on the new grid \code{new_xarg}.
These weights are defined as the product of
\code{object@sign} and \eqn{\hat\beta(x^{new})} (from Step 1).

Predicted functional regression indices
are defined as the product of
\code{object@sign} and \code{intg} (from Step 2).
Multiplication by \code{object@sign} is required to ensure
that the resulting functional regression indices
are positively associated with the functional predictor values
at the selected quantile of \code{object@xarg}.
}
}
\section{Slots}{

\describe{
\item{\code{formula,data,xarg}}{see explanations in section \strong{Arguments}}

\item{\code{gam}}{\link[mgcv]{gam} object, the returned object of helper function \code{\link[=FR_gam]{FR_gam()}}}

\item{\code{sign}}{\link[base]{double} scalar of either 1 or -1,
see Step 5 in section \strong{Details} on function \code{\link[=FRindex]{FRindex()}}}

\item{\code{index,weight}}{\link[base]{double} \link[base]{vector}s,
functional regression indices and functional regression weights, respectively.
See section \strong{Details} on function \code{\link[=FRindex]{FRindex()}}}
}}

\section{Using \code{...} in \code{FRindex()}}{


Function \code{\link[=FRindex]{FRindex()}} passes the parameters
\code{xarg}, \code{family}, \code{knot_pct} and \code{knot.value}
into helper function \code{\link[=FR_gam]{FR_gam()}} through three dots \code{...}.

The most important parameter among them is \code{xarg}.
The default argument of the parameter \code{xarg} comes
from the column names of the \link[base]{matrix} of
tabulated functional predictor values \eqn{X}.
This is particularly convenient when
the functional predictor is the \link[stats]{quantile} function,
and \code{data} is the returned object of function \code{\link[=clusterQp]{clusterQp()}}.

Both \code{\link[=FRindex]{FRindex()}} and helper function \code{\link[=FR_gam]{FR_gam()}} accept user-provided \code{xarg}.
In such case, the provided values will be checked such that
\enumerate{
\item \code{xarg} is a \link[base]{numeric} \link[base]{vector} without missingness
\item \link[base]{length} of \code{xarg} is the same as the number of columns of \link[base]{matrix} \eqn{X}
\item \code{xarg} must be strictly sorted (see \link[base]{is.unsorted})
}
Otherwise, an error message will be returned.
}

\section{Details of Helper Function}{


Helper function \code{\link[=FR_gam]{FR_gam()}} uses \link[mgcv]{gam} to estimate the functional coefficient by fitting functional regression model.
}

\section{Returns of Helper Functions}{

Helper function \code{\link[=FR_gam]{FR_gam()}} returns a \link[mgcv]{gam} object, with additional \link[base]{attributes}

\describe{
\item{\code{attr(,'X')}}{\link[base]{double} \link[base]{matrix} of tabulated functional predictor values \eqn{X}}
\item{\code{attr(,'xarg')}}{\link[base]{double} \link[base]{vector}, see explanation of parameter \code{xarg}}
}
}

\examples{
library(survival)

pt = unique(Ki67$PATIENT_ID)
length(pt) # 622
# set.seed if necessary
train_pt = sample(pt, size = 500L)
Ki67q = clusterQp(Marker ~ ., data = Ki67, exclude = c('tissueID','inner_x','inner_y'))
train_q = subset(Ki67q, PATIENT_ID \%in\% train_pt)
test_q = subset(Ki67q, !(PATIENT_ID \%in\% train_pt))
train_q$Marker = log1p(train_q$Marker)
test_q$Marker = log1p(test_q$Marker)

FRi = FRindex(Surv(RECFREESURV_MO, RECURRENCE) ~ Marker, data = train_q)
FRi@index # functional regression index
FRi@weight # functional regression weights
head(show(FRi)) # append `FRi` to the data

(FRi_test = predict(FRi, newdata = test_q))

FRi_train = predict(FRi)
# stopifnot(identical(FRi@index, c(FRi_train)), 
#  identical(FRi@weight, attr(FRi_train, 'weight')))

# set.seed if necessary
Ki67bbc_v2 = BBC_dichotom(Surv(RECFREESURV_MO, RECURRENCE) ~ NodeSt + Tstage, 
  data = data.frame(train_q, FRi_std = std_IQR(FRi_train)), 
  dichotom = ~ FRi_std)
summary(Ki67bbc_v2)

Ki67q = clusterQp(Marker ~ ., data = Ki67, exclude = c('tissueID','inner_x','inner_y'))
Ki67q$Marker = log1p(Ki67q$Marker)

library(survival)
FR_gam(Surv(RECFREESURV_MO, RECURRENCE) ~ Marker, data = Ki67q)

}
\references{
Cui, E., Crainiceanu, C. M., & Leroux, A. (2021).
Additive Functional Cox Model. Journal of Computational and Graphical Statistics.
\doi{10.1080/10618600.2020.1853550}

Gellar, J. E., Colantuoni, E., Needham, D. M., & Crainiceanu, C. M. (2015).
Cox regression models with functional covariates for survival data. Statistical Modelling.
\doi{10.1177/1471082X14565526}
}
