% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dose-response-analysis.R
\name{growth.drFit}
\alias{growth.drFit}
\title{Perform a dose-response analysis on response vs. concentration data}
\usage{
growth.drFit(
  gcTable,
  control = growth.control(dr.method = "model", dr.model = c("gammadr", "multi2", "LL.2",
    "LL.3", "LL.4", "LL.5", "W1.2", "W1.3", "W1.4", "W2.2", "W2.3", "W2.4", "LL.3u",
    "LL2.2", "LL2.3", "LL2.3u", "LL2.4", "LL2.5", "AR.2", "AR.3", "MM.2"),
    dr.have.atleast = 6, dr.parameter = "mu.linear", nboot.dr = 0, smooth.dr = NULL,
    log.x.dr = FALSE, log.y.dr = FALSE)
)
}
\arguments{
\item{gcTable}{A dataframe containing the data for the dose-response curve estimation. Such table of class \code{gcTable} can be obtained by running \code{\link{growth.gcFit}}.}

\item{control}{A \code{grofit.control} object created with \code{\link{growth.control}}, defining relevant fitting options.}

\item{dr.method}{(Character) Define the method used to perform a dose-responde analysis: smooth spline fit (\code{'spline'}) or model fitting (\code{'model'}).}

\item{dr.model}{(Character) Provide a list of models from the R package 'drc' to include in the dose-response analysis (if \code{dr.method = 'model'}). If more than one model is provided, the best-fitting model will be chosen based on the Akaike Information Criterion.}

\item{dr.have.atleast}{(Numeric) Minimum number of different values for the response parameter one should have for estimating a dose response curve. Note: All fit procedures require at least six unique values. Default: \code{6}.}

\item{dr.parameter}{(Character or numeric) The response parameter in the output table to be used for creating a dose response curve. See \code{\link{growth.drFit}} for further details. Default: \code{'mu.linfit'}, which represents the maximum slope of the linear regression. Typical options include: \code{'mu.linfit'}, \code{'lambda.linfit'}, \code{'dY.linfit'}, \code{'mu.spline'}, \code{'dY.spline'}, \code{'mu.model'}, and \code{'A.model'}.}

\item{smooth.dr}{(Numeric) Smoothing parameter used in the spline fit by smooth.spline during dose response curve estimation. Usually (not necessesary) in (0; 1]. See \code{\link{smooth.spline}} for further details. Default: \code{NULL}.}

\item{log.x.dr}{(Logical) Indicates whether \code{ln(x+1)} should be applied to the concentration data of the dose response curves. Default: \code{FALSE}.}

\item{log.y.dr}{(Logical) Indicates whether \code{ln(y+1)} should be applied to the response data of the dose response curves. Default: \code{FALSE}.}

\item{nboot.dr}{(Numeric) Defines the number of bootstrap samples for EC50 estimation. Use \code{nboot.dr = 0} to disable bootstrapping. Default: \code{0}.}
}
\value{
An object of class \code{drFit}.
\item{raw.data}{Data that passed to the function as \code{gcTable}.}
\item{drTable}{Dataframe containing condition identifiers, fit options, and results of the dose-response analysis.}
\item{drBootSplines}{List of all \code{drBootSpline} objects generated by the call of \code{\link{growth.drBootSpline}} for each distinct experiment.}
\item{drFittedSplines}{List of all \code{drFitSpline} objects generated by the call of \code{\link{growth.drFitSpline}} for each distinct experiment.}
\item{control}{Object of class \code{grofit.control} containing list of options passed to the function as \code{control}.}
}
\description{
\code{growth.drFit} serves to determine dose-response curves on every condition in a
dataset. The response parameter can be chosen from every physiological parameter in a
\code{gcTable} table which is obtained via \code{\link{growth.gcFit}}. \code{\link{growth.drFit}}
calls the functions \code{\link{growth.drFitSpline}} and \code{\link{growth.drBootSpline}}, or \code{\link{growth.drFitModel}} to
generate a table with estimates for EC50 and respecting statistics.
}
\examples{
\donttest{
# Create random growth data set
rnd.data1 <- rdm.data(d = 35, mu = 0.8, A = 5, label = 'Test1')
rnd.data2 <- rdm.data(d = 35, mu = 0.6, A = 4.5, label = 'Test2')

rnd.data <- list()
rnd.data[['time']] <- rbind(rnd.data1$time, rnd.data2$time)
rnd.data[['data']] <- rbind(rnd.data1$data, rnd.data2$data)

# Run growth curve analysis workflow
gcFit <- growth.gcFit(time = rnd.data$time,
                       data = rnd.data$data,
                       parallelize = FALSE,
                       control = growth.control(fit.opt = 's',
                                                suppress.messages = TRUE))

# Perform dose-response analysis
drFit <- growth.drFit(gcTable = gcFit$gcTable,
             control = growth.control(dr.parameter = 'mu.spline'))

# Inspect results
summary(drFit)
plot(drFit)
}
}
\references{
Matthias Kahm, Guido Hasenbrink, Hella Lichtenberg-Frate, Jost Ludwig, Maik Kschischo (2010). \emph{grofit: Fitting Biological Growth Curves with R}. Journal of Statistical Software, 33(7), 1-21. DOI: 10.18637/jss.v033.i07
}
\seealso{
Other growth fitting functions: 
\code{\link{growth.gcBootSpline}()},
\code{\link{growth.gcFitLinear}()},
\code{\link{growth.gcFitModel}()},
\code{\link{growth.gcFitSpline}()},
\code{\link{growth.gcFit}()},
\code{\link{growth.workflow}()}
}
\concept{growth fitting functions}
