% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pairwise.r
\name{pairwise}
\alias{pairwise}
\title{Pairwise comparisons of lm.rrpp fits}
\usage{
pairwise(fit, fit.null = NULL, groups, covariate = NULL,
  print.progress = FALSE)
}
\arguments{
\item{fit}{A linear model fit using \code{\link{lm.rrpp}}.}

\item{fit.null}{An alternative linear model fit to use as a null model for RRPP, if the null model
of the fit is not desired.  Note, for FRPP this argument should remain NULL and FRPP
must be established in the lm.rrpp fit.}

\item{groups}{A factor or vector that is coercible into a factor, describing the levels of
the groups for which to find LS means or slopes.  Normally this factor would be part of the 
model fit, but it is not necessary for that to be the case in order to obtain results.}

\item{covariate}{A numeric vector for which to calculate slopes for comparison  If NULL, 
LS means will be calculated instead of slopes.  Normally this variable would be part of the 
model fit, but it is not necessary for that to be the case in order to obtain results.}

\item{print.progress}{If a null model fit is provided, a logical value to indicate whether analytical 
results progress should be printed on screen.  Unless large data sets are analyzed, this argument 
is probably not helpful.}
}
\value{
An object of class \code{pairwise} is a list containing the following
\item{LS.means}{LS means for groups, across permutations.}
\item{slopes}{Slopes for groups, across permutations.}
\item{means.dist}{Pairwise distances between means, across permutations.}
\item{means.vec.cor}{Pairwise vector correlations between means, across permutations.}
\item{slopes.lengths}{Slope lengths, by group, across permutations.}
\item{slopes.dist}{Pairwise distances between slope lengths, across permutations.}
\item{slopes.vec.cor}{Pairwise vector correlations between slope vectors, across permutations.}
\item{n}{Sample size}
\item{p}{Data dimensions; i.e., variable number}
\item{PermInfo}{Information for random permutations, passed on from lm.rrpp fit and possibly
modified if an alternative null model was used.}
}
\description{
Function generates distributions of pairwise statistics for a lm.rrpp fit and
returns important statistics for hypothesis tests.
}
\details{
Based on an lm.rrpp fit, this function will find fitted values over all permutations and based 
on a grouping factor, calculate either least squares (LS) means or slopes, and pairwise statistics among
them.  Pairwise statistics have two flavors: distances and vector correlations (or angles).  The distance
statistics calculate either the length of vectors between LS mean vectors or the absolute difference between 
slope vector lengths.  The vector correlations are the inner product of vectors that have been transformed to unit length.
The arccosine (acos) of this value is the angle between vectors, which can be expressed in radians or degrees, and is
used as a test statistic (with the null hypothesis that vectors are parallel; angle = 0).
Over all permutations, these values can be calculated to generate random distributions using the null model.  The 
null model is defined via \code{\link{lm.rrpp}}, but one can also use an alternative null model as an optional argument.
In this case, residual randomization in the permutation procedure (RRPP) will be performed using the alternative null model 
to generate fitted values.  If full randomization of values (FRPP) is preferred,
it must be established in the lm.rrpp fit and an alternative model should not be chosen.

Observed statistics, effect sizes, P-values, and one-tailed confidence limits based on the confidence requested will
be summarized with the \code{\link{summary.pairwise}} function.  The \code{\link{summary.pairwise}} function will allow one
to select between distance or vector correlation tests, whether angles are measured in radians or degrees, and the level of
confidence for the test.  Confidence limits are inherently one-tailed as
the statistics are similar to absolute values.  For example, a distance is analogous to an absolute difference.  Therefore,
the one-tailed confidence limits are more akin to two-tailed hypothesis tests.  (A comparable example is to use the absolute 
value of a t-statistic, in which case the distribution has a lower bound of 0.)

If data are univariate, test.type = 'cor' should not be chosen because the vector correlation between univariate 
vectors is always 1.  Rather, cor.type = 'dist' will return the absolute difference between slopes or between means.  
Please note that this function will generate results if test.type = 'cor' for univariate data, but the results will 
not make much sense.
}
\examples{

# Examples use geometric morphometric data on pupfishes
# See the package, geomorph, for details about obtaining such data

# Body Shape Analysis (Multivariate)----------------------------------------------------

data("Pupfish")

# Note:

dim(Pupfish$coords) # highly multivariate!

Pupfish$logSize <- log(Pupfish$CS) # better to not have functions in formulas

# Note: one should use all dimensions of the data but with this example, there are many
# Thus, only three principal components will be used for demonstration purposes.

Pupfish$Y <- prcomp(Pupfish$coords)$x[, 1:3]

# Pairwise comparisons of LS means

fit1 <- lm.rrpp(Y ~ logSize + Sex * Pop, SS.type = "I", 
data = Pupfish, print.progress = FALSE, iter = 999) 
summary(fit1, formula = FALSE)
anova(fit1) 

pup.group <- interaction(Pupfish$Sex, Pupfish$Pop)
pup.group
PW1 <- pairwise(fit1, groups = pup.group)
PW1
summary(PW1, confidence = 0.95, test.type = "dist") # distances between means
summary(PW1, confidence = 0.95, test.type = "dist", stat.table = FALSE)
summary(PW1, confidence = 0.95, test.type = "VC", 
   angle.type = "deg") # correlation between mean vectors (angles in degrees)

# Pairwise comparisons of slopes

fit2 <- lm.rrpp(Y ~ logSize * Sex * Pop, SS.type = "I", 
data = Pupfish, print.progress = FALSE, iter = 999) 
summary(fit2, formula = FALSE)
anova(fit1, fit2)

# Using a null fit that excludes all factor-covariate interactions, not just the last one  

PW2 <- pairwise(fit2, fit.null = fit1, groups = pup.group, covariate = Pupfish$logSize) 
PW2
summary(PW2, confidence = 0.95, test.type = "dist") # distances between slope vector lengths
summary(PW2, confidence = 0.95, test.type = "dist", stat.table = FALSE)
summary(PW2, confidence = 0.95, test.type = "VC",
   angle.type = "deg") # correlation between slope vectors (and angles)
}
\references{
Collyer, M.L., D.J. Sekora, and D.C. Adams. 2015. A method for analysis of phenotypic change for phenotypes described
by high-dimensional data. Heredity. 115:357-365.

Adams, D.C and M.L. Collyer. 2018. Multivariate phylogenetic anova: group-clade aggregation, biological 
challenges, and a refined permutation procedure. Evolution. In press.
}
\seealso{
\code{advanced.procD.lm} within \code{geomorph}; \code{\link{lm.rrpp}} for model fits
}
\author{
Michael Collyer
}
\keyword{analysis}
