#' @title Write Raven GridWeights file
#'
#' @description
#' rvn_gridweights_write is used to write a Raven grid weights file.
#'
#' @details
#' This function write the gridweights to the specified grid weights text file.
#'
#' @param gridweights list with the number of HRUs, number of grid cells, and gridweights data frame
#' @param outfile file to write gridweights to
#' @return \item{hyd}{data frame from the file with standardized names}
#' @seealso \code{\link{rvn_gen_gridweights}} for generating a grid weights file from an HRU shapefile
#' and netcdf file.
#'
#' @examples
#'
#' # load example rvh file
#' nith <- system.file("extdata",'Nith.rvh', package = "RavenR")
#' rvh <- rvn_rvh_read(nith)
#'
#' # adjust HRU shapefile to one per subbasin for demonstration
#' rvh$HRUtable <- rvh$HRUtable[c(1,6,15,25),]
#' rvh$HRUtable$Area <- rvh$SBtable$Area
#' rvh$HRUtable$ID <- rvh$HRUtable$SBID
#'
#' # define HRU shapefile path (use subbasin shapefile for example)
#' hrushpfile <- system.file("extdata","Nith_shapefile_sample.shp",package = "RavenR")
#'
#' # get grid shapefile from netcdf file
#' nithnc <- system.file("extdata/Nith_era5_sample.nc", package="RavenR")
#' gridshp <- rvn_netcdf_to_gridshp(ncfile=nithnc, projID=26917)
#'
#' # calculate gridweights
#' gw <- rvn_gen_gridweights(hrushpfile, gridshp,
#' gridIDcol = 'GridIDs', HRUIDcol = "subID")
#'
#' # write the gridweights (gw) object to file
#' tfout <- file.path(tempdir(), "Nith_GridWeights.rvt")
#' rvn_gridweights_write(gw, outfile=tfout)
#'
#' @export rvn_gridweights_write
rvn_gridweights_write <- function(gridweights=NULL, outfile="GridWeights.rvt") {

  if (missing(gridweights)) {
    stop("Requires a gridweights object.")
  }

  ## additional checks on integrity of gridweights object
  ## xxx


  # write Gridweights file
  #----------------------------------------------------------

  dfgrid <- gridweights$GridWeights
  numhrus <- gridweights$NumberHRUs
  numgrids <- gridweights$NumberGridCells


  # print(sprintf("Writing %s", outfile))

  write("# -------------------------------------------------",append=FALSE,file=outfile)
  write("# grid weights file generated by RavenR gridweights.gen.R utility",append=TRUE,file=outfile)
  write("# -------------------------------------------------",append=TRUE,file=outfile)
  write(":GridWeights",append=TRUE,file=outfile)
  write(paste("  :NumberHRUs ", numhrus),append=TRUE,file=outfile)
  write(paste("  :NumberGridCells ", numgrids),append=TRUE,file=outfile)

  for (i in 1:nrow(dfgrid)) {
    write(sprintf("   %i\t%i\t%.8f",
                  dfgrid$HRUID[i], dfgrid$GRIDID[i], dfgrid$WEIGHT[i]),
          append=TRUE,file=outfile)
    # write(paste("  ",HRUID," ",GridID ," ",wt),append=TRUE,file=outfile)
  }

  write(":EndGridWeights",append=TRUE,file=outfile)
  print(sprintf("Finished writing to %s", outfile))

    return(TRUE)
}
