#' Bootstrapping - case resampling
#'
#' Generate replicates of the original data using random sampling with replacement.
#' Population parameters are then estimated from each replicate.
#' @param project Monolix project 
#' @param nboot [optional] number of bootstrap replicates (default=100)
#' @param dataFolder [optional] folder where already generated datasets are stored, e.g dataFolder="./dummy_project/boot/" (default: data set are generated by bootmlx)
#' @param parametric [optional] boolean to define if parametric bootstrap is performed (new data is drawn from the model), (default: false)
#' @param tasks [optional] vector of booleans defining the list of tasks to perform (default: estimation of the population parameters)
#' available tasks: populationParameterEstimation, conditionalDistributionSampling,
#' conditionalModeEstimation, standardErrorEstimation, logLikelihoodEstimation, plots  
#' @param settings [optional] a list of settings for the resampling and the results:
#' \itemize{
#' \item \code{N} the number of individuals in each bootstrap data set 
#' (default value is the  number of individuals in the original data set).
#' \item \code{newResampling} boolean to generate the data sets again if they already exist (default=FALSE).
#' \item \code{covStrat} a categorical covariate of the project. The original distribution of this covariate
#' is maintained in each resampled data set if covStrat is defined (default=NULL). Notice that if the categorical covariate is varying 
#' within the subject (in case of IOV), it will not be taken into account.
#' \item \code{plot} boolean to choose if the distribution of the bootstraped esimates is displayed
#' (default = FALSE)
#' \item \code{level} level of the bootstrap confidence intervals of the population parameters
#' (default = 0.90)
#' \item \code{seed} seed for the generation of the data sets (default = NA)
#' }
#' @return a data frame with the bootstrap estimates
#' @examples
#' \dontrun{
#' # RsmlxDemo1.mlxtran is a Monolix project for modelling the PK of warfarin using a PK model 
#' # with parameters ka, V, Cl.
#' 
#' # In this example, bootmlx will generate 100 random replicates of the original data and will
#' # use Monolix to estimate the population parameters from each of these 100 replicates:
#' r1 <- bootmlx(project="RsmlxDemo1.mlxtran")
#'   
#' # 5 replicates will now be generated, with 50 individuals in each replicate:
#' r2 <- bootmlx(project="RsmlxDemo1.mlxtran",  nboot = 5, settings = list(N = 50))
#' 
#' # Proportions of males and females in the original dataset will be preserved   
#' # in each replicate:
#' r3 <- bootmlx(project="RsmlxDemo1.mlxtran",  settings = list(covStrat = "sex"))
#' }
#' 
#' # See http://rsmlx.webpopix.org/userguide/bootmlx/ for detailed examples of use of bootmlx
#' # Download the demo examples here: http://rsmlx.webpopix.org/installation
#' 
#' 
#' @importFrom graphics boxplot lines par plot
#' @importFrom stats quantile
#' @export
bootmlx <- function(project, nboot = 100, dataFolder = NULL, parametric = FALSE,
                    tasks=c(populationParameterEstimation=TRUE), 
                    settings = NULL){

  params <- as.list(match.call(expand.dots=T))[-1]
  
  monolixPath <- mlx.getLixoftConnectorsState()$path
  RsmlxDemo1.project <- RsmlxDemo2.project <- warfarin.data  <- resMonolix <- NULL
  
  r <- prcheck(project, f="boot", settings=settings)
  if (r$demo)
    return(r$res)
  project <- r$project
  
  mlx.loadProject(project)
  exportDir <- mlx.getProjectSettings()$directory
  projectName <- basename(tools::file_path_sans_ext(project))
  
  # Check and initialize inputs ------------------------------------------------
  if (is.element("dataFolder", names(params)) & (is.element("nboot", names(params)))) {
    stop("DataFolder and nBoot can not be used both at the same time.", call.=F)
  }

  .check_strict_pos_integer(nboot, "nboot")
  if (is.null(nboot)) nboot <- 100
  .check_bool(parametric, "parametric")
  .check_in_vector(names(tasks), "tasks names", names(mlx.getScenario()$tasks))
  .check_bool(tasks, "tasks")
  tasks['populationParameterEstimation'] <- TRUE

  # check and initialize the settings
  settings <- .initBootstrapSettings(settings, parametric)
  settings$nboot <- nboot
  plot.res <- settings$plot
  settings$plot<- NULL
  level <- settings$level
  
  settings$level<- NULL
  
  if (!is.null(dataFolder)) {
    dataFiles <- list.files(path = dataFolder, pattern = '*.txt|*.csv')
    if (length(dataFiles) > 0) {
      settings$nboot <- length(dataFiles)
      settings$newResampling <- FALSE
    } else {
      stop("Folder ", dataFolder, " does not exist or does not contain any data set.", call.=F)
    }
    
    dataFolderToUse = dataFolder
    boot.folder = '/bootstrap/'
    if (!dir.exists(file.path(exportDir, boot.folder))) {
      dir.create(file.path(exportDir, boot.folder))
    }
  } else {
    # Prepare all the output folders

    # generate data sets from the initial data set
    if (parametric) {
      boot.folder = '/bootstrap/parametric/'
    } else {
      boot.folder = '/bootstrap/nonParametric/'
    }
    if (settings$newResampling) {
      cleanbootstrap(project, boot.folder)
    }
    dataFolderToUse = file.path(exportDir, boot.folder, 'data')

    if (parametric) {
      version <- mlx.getLixoftConnectorsState()$version
      v <- regmatches(version, regexpr("^[0-9]*", version, perl = TRUE))
      if (v >= 2020) {
        generateDataSetParametricSimulx(project, settings, boot.folder)
      } else {
        generateDataSetParametricMlxR(project, settings, boot.folder)
      }
    } else {
      generateDataSetResample(project, settings, boot.folder)
    }
    useLL = F
  }
  generateBootstrapProject(project, dataFolder = dataFolderToUse, boot.folder=boot.folder)
  
  paramResults <- NULL 
  # for (indexSample in seq_len(settings$nboot)) {
  #   projectBoot <-  paste0(exportDir, boot.folder, projectName, '_bootstrap_', toString(indexSample), '.mlxtran')
  #   mlx.loadProject(projectBoot)
  #   cat(paste0('Project ', toString(indexSample), '/', toString(settings$nboot)))
  #   
  #   # Check if the run was done
  #   # if(!file.exists(paste0(mlx.getProjectSettings()$directory,'/populationParameters.txt'))){
  #   launched.tasks <- mlx.getLaunchedTasks()
  #   g <- mlx.getScenario()
  #   g$tasks <- tasks
  #   mlx.setScenario(g)
  #   scenario.tasks <- mlx.getScenario()$tasks
  #   mlx.saveProject(projectBoot)
  #   if (!launched.tasks[["populationParameterEstimation"]]) {
  #     if (sum(scenario.tasks)==1)
  #       cat(' => Estimating the population parameters \n')
  #     else
  #       cat(' => Running the scenario \n')
  #     mlx.runScenario()
  #   } else {
  #     missing.tasks <- 0
  #     if (scenario.tasks[['conditionalDistributionSampling']] && !launched.tasks[['conditionalDistributionSampling']]) {
  #       missing.tasks <- missing.tasks + 1 
  #       if (missing.tasks==1) cat(' => Running the missing tasks \n')
  #       mlx.runConditionalDistributionSampling()
  #     }
  #     if (scenario.tasks[['conditionalModeEstimation']] && !launched.tasks[['conditionalModeEstimation']]) {
  #       missing.tasks <- missing.tasks + 1 
  #       if (missing.tasks==1) cat(' => Running the missing tasks \n')
  #       mlx.runConditionalModeEstimation()
  #     }
  #     if (scenario.tasks[['standardErrorEstimation']] && launched.tasks[['standardErrorEstimation']]==FALSE) {
  #       missing.tasks <- missing.tasks + 1 
  #       if (missing.tasks==1) cat(' => Running the missing tasks \n')
  #       mlx.runStandardErrorEstimation(linearization=g$linearization)
  #     }
  #     if (scenario.tasks[['logLikelihoodEstimation']] && !launched.tasks[['logLikelihoodEstimation']]) {
  #       missing.tasks <- missing.tasks + 1 
  #       if (missing.tasks==1) cat(' => Running the missing tasks \n')
  #       mlx.runLogLikelihoodEstimation()
  #     }
  #     if (missing.tasks==0)  {
  #       if (sum(scenario.tasks)==1)
  #         cat(' => Population parameters already estimated \n')
  #       else
  #         cat(' => Tasks already performed \n')
  #     }
  #   }
  #   
  #   paramResults <-  rbind(paramResults, mlx.getEstimatedPopulationParameters())
  # }
  # colnames(paramResults) <- names(mlx.getEstimatedPopulationParameters())
  # paramResults <- as.data.frame(paramResults)
  # 
  # res.file <- file.path(exportDir,boot.folder,"populationParameters.txt")
  # write.table(x = paramResults, file = res.file,
  #             eol = "\n", sep = ",", col.names = TRUE, quote = FALSE, row.names = FALSE)
  # cat("Estimated population parameters have been saved: ", normalizePath(res.file), ".\n")
  # 
  # # Plot the results
  # if (plot.res) {
  #   nbFig <- ncol(paramResults)
  #   x_NbFig <- ceiling(max(sqrt(nbFig),1)); y_NbFig <- ceiling(nbFig/x_NbFig)
  #   par(mfrow = c(x_NbFig, y_NbFig), oma = c(0, 3, 1, 1), mar = c(3, 1, 0, 3), mgp = c(1, 1, 0), xpd = NA)
  #   for(indexFigure in 1:nbFig){
  #     res <- paramResults[,indexFigure]
  #     resQ <- quantile(res,c((1-level)/2,(1+level)/2))
  #     bxp <- boxplot(res, xlab = paste0(colnames(paramResults)[indexFigure],'\n',level*100,'% CI: [',toString(round(resQ[1],3)),', ',toString(round(resQ[2],3)),']'))
  #   }
  # }
  return(paramResults)
}

##############################################################################
# Generate data sets by resampling the original data set
##############################################################################
generateDataSetResample = function(project, settings, boot.folder){
  
  if (!file.exists(project)) {
    stop("project '", project, "' does not exist", call.=F)
  }
  
  mlx.loadProject(project)   
  
  if(is.null(settings)){
    settings$nboot <- 100 
    settings$N <- NA
    settings$covStrat <- NA
    settings$seed <- NA
  }
  if(!is.na(settings$seed)){set.seed(settings$seed)}
  
  # Prepare all the output folders
  exportDir <- mlx.getProjectSettings()$directory
  projectName <- basename(tools::file_path_sans_ext(project))
  dir.create(file.path(exportDir, boot.folder), showWarnings = FALSE, recursive = TRUE)
  
  # Get the data set information
  referenceDataset <- mlx.getData()
  cov <- mlx.getCovariateInformation()
  datasetFile <- referenceDataset$dataFile
  refCovInfo  <- mlx.getCovariateInformation()
  # Get the index in mlx.getCovariateInformation()$ of the covariates used in the statistical model
  indexUsedCat <- NULL
  for(indexCov in seq_len(length(refCovInfo$name))){
    if(grepl("categorical", refCovInfo$type[indexCov], fixed=TRUE)){
      # Is the covariate used in the covariate model
      isUsed <- F
      idCov <- which(names(mlx.getIndividualParameterModel()$covariateModel[[1]])==refCovInfo$name[indexCov])
      for(indexParam in 1:length(mlx.getIndividualParameterModel()$covariateModel)){
        isUsed <- isUsed||mlx.getIndividualParameterModel()$covariateModel[[indexParam]][idCov] 
      }
      if(isUsed){
        indexUsedCat <- c(indexUsedCat, indexCov)
      }
    }
  }
  
  cat("Generating data sets with initial data set resampling...\n")
  dir.create(file.path(exportDir, boot.folder, 'data'), showWarnings = FALSE)
  
  # Load the data set
  sepBoot <- .getDelimiter(datasetFile)
  dataset <- read.table(file=datasetFile, sep=sepBoot, header=T, dec=".", check.names=FALSE)
  names(dataset) <- gsub(" ", "_", names(dataset))
  
  indexID <- which(referenceDataset$headerTypes=="id")
  nameID <- unique(dataset[, indexID])
  nbIndiv <- length(nameID)
  if(is.na(settings[['N']])){settings[['N']] = nbIndiv}
  
  validID <- list()
  
  if(is.na(settings$covStrat)){
    nbCAT = 1
    indexPropCAT <- 1
    propCAT <- rep(settings[['N']], nbCAT)
    validID[[indexPropCAT]] <- nameID
  }else{
    indexCAT <- which(names(cov$covariate) == settings$covStrat)
    
    isCatVaryID <- F
    for(indexIDtestCAT in 1:length(nameID)){
      cat <- cov$covariate[which(cov$covariate[,1]==nameID[indexIDtestCAT]),indexCAT]
      if(length(unique(cat))>1){
        isCatVaryID <- T
      }
    }
    
    if(isCatVaryID){# The covariate vary within the subject occasion, 
      cat(paste0("The generated data set can not preserve proportions of ", settings$covStrat," as the covariate vary in within the subject.\n"))
      nbCAT = 1
      indexPropCAT <- 1
      propCAT <- rep(settings[['N']], nbCAT)
      validID[[indexPropCAT]] <- nameID
    }else{
      catValues <- cov$covariate[,indexCAT]
      nameCAT <- unique(catValues)
      nbCAT <- length(nameCAT)
      propCAT <- rep(settings[['N']], nbCAT)
      validID <- list()
      for(indexPropCAT in 1:nbCAT){
        indexIdCat <- which(catValues==nameCAT[indexPropCAT])
        propCAT[indexPropCAT] <- max(1,floor(settings[['N']]*length(indexIdCat)/nbIndiv))
        validID[[indexPropCAT]] <- as.character(cov$covariate[indexIdCat,1])
      }
    }
  }
  warningAlreadyDisplayed <- F
  
  for(indexSample in 1:settings$nboot){
    datasetFileName <- file.path(exportDir,boot.folder,'data', paste0('dataset_',toString(indexSample),'.csv'))
    if(!file.exists(datasetFileName)){
      ##################################################################################################################
      # Generate the data set
      ##################################################################################################################
      # Sample the IDs
      areAllModalitiesdrawn <- F
      while(!areAllModalitiesdrawn){
        sampleIDs <- NULL
        for(indexValidID in 1:length(validID)){
          if(length(validID[[indexValidID]])==1){
            sampleIDs <- c(sampleIDs,  rep(x = validID[[indexValidID]], times = propCAT[indexValidID]) )
          }else{
            samples <- NULL
            samples <- sample(x = validID[[indexValidID]], size = propCAT[indexValidID], replace = TRUE)
          }
          sampleIDs <- c(sampleIDs, as.character(samples))
        }
        areAllModalitiesdrawn <- T
        if(length(indexUsedCat)>0){
          # Check if all used modalities are in the data set
          for(iUsedCat in 1:length(indexUsedCat)){
            catModalities <- unique(refCovInfo$covariate[,indexUsedCat[iUsedCat]+1])
            catSamplesModalities <- NULL
            for(indexSamples in 1:length(sampleIDs)){
              idIndex = which(refCovInfo$covariate[,1]==sampleIDs[indexSamples])
              catSamplesModalities <- c(catSamplesModalities, refCovInfo$covariate[idIndex,indexUsedCat[iUsedCat]+1])
            }
            areAllModalitiesdrawn <- areAllModalitiesdrawn&(length(catModalities)==length(unique(catSamplesModalities)))
          }
        }
      }
      
      if(!(length(sampleIDs)==settings[['N']])){
        if(!warningAlreadyDisplayed){
          cat(paste0("The generated data set contains only ",length(sampleIDs)," individuals because otherwise categorical proportions of ",settings$covStrat," cannot be kept.\n"))
          warningAlreadyDisplayed = TRUE
        }
      }
      # Get the datas
      data <- NULL
      dataID <- NULL
      indexLineFull <- NULL
      for(indexSampleSize in 1:length(sampleIDs)){
        indexLine <- which(dataset[,indexID]==sampleIDs[indexSampleSize])
        indexLineFull <- c(indexLineFull,indexLine)
        dataID <- c(dataID, rep(indexSampleSize,length(indexLine)))
      }
      data <- dataset[indexLineFull,]
      data[,indexID] <- dataID
      write.table(x = data, file = datasetFileName, sep = sepBoot,
                  eol = '\n', quote = FALSE, dec = '.',  row.names = FALSE, col.names = TRUE )
    }
  }
}

##############################################################################
# Generate data sets by resimulating the project
##############################################################################
generateDataSetParametricMlxR= function(project, settings=NULL, boot.folder=NULL){
  
  if(!file.exists(project)){
    stop("project '", project, "' does not exist.", call.=F)
  }
  suppressMessages(mlx.initializeLixoftConnectors())
  mlx.loadProject(project)   
  # Prepare all the output folders
  exportDir <- mlx.getProjectSettings()$directory
  projectName <- basename(tools::file_path_sans_ext(project))
  dir.create(file.path(exportDir, boot.folder), showWarnings = FALSE, recursive = TRUE)
  dir.create(file.path(exportDir, boot.folder, 'data'), showWarnings = FALSE, recursive = TRUE)
  if(is.null(settings)){
    settings$nboot <- 100 
    settings$N <- NA
    settings$covStrat <- NA
    settings$seed <- NA
  }
  if(!is.na(settings$seed)){set.seed(settings$seed)}
  
  monolixPath <- mlx.getLixoftConnectorsState()$path
  mlx.initializeLixoftConnectors(software="simulx", path=monolixPath, force=TRUE)
  cat("Generating data sets...\n")
  
  for(indexSample in 1:settings$nboot){
    #suppressMessages(mlx.initializeLixoftConnectors())
    
    datasetFileName <- paste0(exportDir,boot.folder, 'data/dataset_',toString(indexSample),'.csv')
    if(!file.exists(datasetFileName)){
      if(!is.na(settings$seed)){
        simSettings = list(format.original=TRUE, seed = settings$seed+indexSample)
      }else{
        simSettings = list(format.original=TRUE)  
      }
      if(!file.exists(datasetFileName)){
  #      res <- simulx(project = project,  result.file=datasetFileName, setting=simSettings)
        res <- mlxR::simulx(project = project,  result.file=datasetFileName, setting=simSettings)
      }
    }
  }
  suppressMessages(mlx.initializeLixoftConnectors())
  
}

generateDataSetParametricSimulx = function(project, settings=NULL, boot.folder=NULL){
  version <- mlx.getLixoftConnectorsState()$version

  if(!file.exists(project)){
    stop("Project ", project, ", does not exist.", call.=F)
  }
  suppressMessages(mlx.initializeLixoftConnectors())
  mlx.loadProject(project)
  obsInfo <- mlx.getObservationInformation()
  mapObservation <- c(obsInfo$mapping)
  
  # return an error in case of regressor
  if (is.element("regressor", mlx.getData()$headerTypes) & version == "2020R1") {
    stop("Projects with regressor are not supported for Boostrap in monolix suite 2020R1.", call. = FALSE)
  }
  # return a warning in case of censoring data
  if (is.element("cens", mlx.getData()$headerTypes)) {
    warning("Note that censoring is not performed on boostrap simulations.", call. = FALSE)
  }
  # Prepare all the output folders
  exportDir <- mlx.getProjectSettings()$directory
  projectName <- basename(tools::file_path_sans_ext(project))
  dir.create(file.path(exportDir, boot.folder), showWarnings = FALSE, recursive = TRUE)
  dir.create(file.path(exportDir, boot.folder, 'data'), showWarnings = FALSE, recursive = TRUE)

  if (!is.na(settings$seed)) smlx.setProjectSettings(settings$seed)
  
  monolixPath <- mlx.getLixoftConnectorsState()$path
  suppressMessages(mlx.initializeLixoftConnectors(software="simulx", path=monolixPath, force=TRUE))
  
  datasetFileName <- file.path(exportDir, boot.folder, "data", "dataset.csv")
  files <- paste0(rep(tools::file_path_sans_ext(datasetFileName), settings$nboot),
                  "_", seq_len(settings$nboot), ".csv")
  # create smlx project - run simulation - save new dataset
  if (!all(file.exists(files))) {
    cat("Generating data sets...\n")
    tryCatch(
      {
        # activate only lixoft errors
        op <- get_lixoft_options()
        set_options(errors = TRUE, warnings = FALSE, info = FALSE)

        smlx.importMonolixProject(project)
        smlx.setNbReplicates(settings$nboot)
        smlx.runSimulation()
        
        writeDataSmlx(filename = datasetFileName, mapObservation = mapObservation)

        set_options(errors = op$errors, warnings = op$warnings, info = op$info)
      },
      message = function(m) {
        m <- as.character(m)
        set_options(errors = op$errors, warnings = op$warnings, info = op$info)
        if (grepl("[ERROR]", as.character(m))) {
          message <- gsub(": ", "", regmatches(m, regexpr(": .*", m, perl=TRUE)))
          stop("[Simulx Error] ", message, call. = FALSE)
        }
      }
    )
      
    # warning for simulx bug
    .catchSimulxBug()
  }
    
  suppressMessages(mlx.initializeLixoftConnectors(software = "monolix"))
  
}

##############################################################################
# Generate projects based on dataFolder
##############################################################################
generateBootstrapProject = function(project, dataFolder, boot.folder){
  
  dataFiles <- list.files(path = dataFolder, pattern = '*.txt|*.csv')
  # Get the data set information
  mlx.loadProject(project)
  exportDir <- mlx.getProjectSettings()$directory
  projectName <- basename(tools::file_path_sans_ext(project))
  
  # 
  scenario = mlx.getScenario()
  scenario$tasks = c(populationParameterEstimation=TRUE)
  mlx.setScenario(scenario)

  # generate boot headers
  referenceDataset <- mlx.getData()
  b1 <- paste0(dataFolder,'/',dataFiles[1])
  bootData <- .getBootData(referenceDataset, b1)

  cat("Generating projects with bootstrap data sets...\n")
  for(indexSample in seq_along(dataFiles)){
    projectBootFileName =  file.path(exportDir, boot.folder, paste0(projectName, '_bootstrap_', toString(indexSample), '.mlxtran'))
    if(!file.exists(projectBootFileName)){
      # deactivate lixoft warnings
      op <- get_lixoft_options()
      set_options(warnings = FALSE)

      bootData$dataFile <- paste0(dataFolder,'/',dataFiles[indexSample])
      mlx.setData(bootData)
      
      # reactivate lixoft warnings
      set_options(warnings = op$warnings)
      
#      mlx.setStructuralModel(modelFile=mlx.getStructuralModel())
      mlx.saveProject(projectFile =projectBootFileName)
    }
  }
}


##################################################################################################################
# Clean the bootstrap folder
##################################################################################################################
cleanbootstrap <- function(project,boot.folder){
  # Prepare all the output folders
  cat('Clearing all previous results and projects \n')
  mlx.loadProject(project)
  exportDir <- mlx.getProjectSettings()$directory
  listProjectsToDelete <- list.files(path = paste0(exportDir,boot.folder), pattern = '*.mlxtran')
  
  if(length(listProjectsToDelete)>0){
    for(indexProject in 1:length(listProjectsToDelete)){
      projectBoot <-  paste0(exportDir,boot.folder,listProjectsToDelete[indexProject])
      exportDirToClean <- gsub(x = projectBoot, pattern = '.mlxtran', '')
      unlink(exportDirToClean, recursive = TRUE)
      unlink(projectBoot, recursive = FALSE)
    }
  }
  unlink(file.path(exportDir, boot.folder,'data'), recursive = TRUE, force = TRUE)
}


###################################################################################
.initBootstrapSettings = function(settings, parametric){
  if (is.null(settings)) settings <- list()

  invalidParametricSettings <- intersect(c("N", "covStrat"), names(settings))
  if (parametric == T & length(invalidParametricSettings)) {
    if (length(invalidParametricSettings) > 1) {
      message <- paste0(paste(invalidParametricSettings, collapse = ", "), " arguments are ignored.")
    } else {
      message <- paste0(invalidParametricSettings, " argument is ignored.")
    }
    warning("In case of parametric boostrap, ", message, call. = FALSE)
    settings <- settings[! names(settings) %in% c("N", "covStrat")]
  }

  .check_in_vector(
    names(settings), "settings",
    c("plot", "level", "N", "newResampling", "covStrat", "seed")
  )
  
  if (!is.element("plot", names(settings))) settings$plot <- F
  if (!is.element("level", names(settings))) settings$level <- 0.90
  if (!is.element("N", names(settings))) settings$N <- NA
  if (!is.element("newResampling", names(settings))) settings$newResampling <- F
  if (!is.element("covStrat", names(settings))) settings$covStrat <- NA
  if (!is.element("seed", names(settings))) settings$seed <- NA
  
  if (!is.na(settings$N)) .check_strict_pos_integer(settings$N, "settings N")
  .check_double(settings$level, "settings level")
  .check_in_range(settings$level, "settings level", 0, 1)
  .check_bool(settings$newResampling, "settings newResampling")
  .check_bool(settings$plot, "settings plot")
  if (!is.na(settings$covStrat)) {
    .check_string(settings$covStrat, "settings covStrat")
    covariates <- mlx.getCovariateInformation()
    catcovariates <- names(covariates$type[covariates$type %in% c("categorical", "categoricaltransformed")])
    .check_in_vector(settings$covStrat, "settings covStrat", catcovariates)
  }
  if (!is.na(settings$seed)) .check_strict_pos_integer(settings$seed, "settings seed")

  if (parametric) settings <- settings[! names(settings) %in% c("N", "covStrat")]
  return(settings)
}

.getBootData <- function(refData, bootFile, smlxHeaders = NULL) {
  bootData <- refData
  mlxHeaders <- refData$header
  mlxHeadersType <- refData$headerTypes
  
  df <- utils::read.table(file = bootFile, nrow = 0, sep = .getDelimiter(bootFile), header = T)
  
  if (is.null(smlxHeaders)) {
    smlxHeaders <- names(df)
  }

  intersectHeaders <- intersect(smlxHeaders, mlxHeaders)
  smlxHeadersType <- rep(NA, length(smlxHeaders))
  for (h in intersectHeaders) {
    smlxHeadersType[smlxHeaders == h] <- mlxHeadersType[mlxHeaders == h]
  }

  matchSmlx <- c(id = "id", amt = "amount", time = "time", tinf = "tinf", OCCevid = "occ",
                 admtype = "admid", ytype = "obsid", evid = "evid", y = "observation")
  smlxHeadersType[smlxHeaders %in% setdiff(smlxHeaders, intersectHeaders)] <- unname(matchSmlx[setdiff(smlxHeaders, intersectHeaders)])
  
  obsInfo <- mlx.getObservationInformation()
  if (any(is.na(smlxHeadersType))) {
    smlxHeadersType[is.na(smlxHeadersType) & smlxHeaders %in% obsInfo$name] <- "observation"
  }
  obsNames <- smlxHeaders[smlxHeadersType == "observation"]
  bootData$dataFile <- bootFile
  bootData$headerTypes <- smlxHeadersType
  
  if ("obsid" %in% mlxHeadersType) {
    if (! "obsid" %in% smlxHeadersType) {
      obsmapping <- obsInfo$mapping[obsNames]
      names(bootData$observationTypes)[names(bootData$observationTypes) == obsmapping] <- obsNames
      bootData$observationTypes <- bootData$observationTypes[names(bootData$observationTypes) %in% obsNames]
      bootData$observationTypes <- unname(bootData$observationTypes)
    } else {
      obsids <- sort(unique(df[[smlxHeaders[smlxHeadersType == "obsid"]]]))
      bootData$observationTypes <- refData$observationTypes[intersect(sort(obsids), names(refData$observationTypes))]
    }
  } else {
    bootData$observationTypes <- unname(refData$observationTypes)
  }

  # warning for simulx bug
  version <- mlx.getLixoftConnectorsState()$version
  if (length(obsInfo$name) > 1 & ! is.element("obsid", smlxHeadersType) & version == "2020R1") {
    warning("Due to a bug in Simulx 2020R1, non-continuous outputs will be excluded from the simulation.", call. = FALSE)
  }
  bootData <- bootData[c("dataFile", "headerTypes", "observationTypes")]

  return(bootData)
}

.catchSimulxBug <- function() {
  # warning for simulx bug
  group <- smlx.getGroups()[[1]]
  ids <- list()
  if (length(grep("^mlx_", group$output)) >= 1) {
    outputelements <- smlx.getOutputElements()[grep("^mlx_", group$output, value = TRUE)]
    outputelements <- outputelements[sapply(outputelements, function(o) is.element("id", names(o$data)))]
    if (length(outputelements) > 1) {
      ids <- c(ids, sapply(outputelements, function(o) unique(o$data$id)))
    } else if (length(outputelements) == 1) {
      ids[[names(outputelements)]] <- unique(outputelements[[1]]$data$id)
    }
  }
  if (any(grepl("^mlx_", group$treatment))) {
    treatelements <- smlx.getTreatmentElements()[grep("^mlx_", group$treatment, value = TRUE)]
    treatelements <- treatelements[sapply(treatelements, function(o) is.element("id", names(o$data)))]
    if (length(treatelements) > 1) {
      ids <- c(ids, sapply(treatelements, function(o) unique(o$data$id)))
    } else if (length(treatelements) == 1) {
      ids[[names(treatelements)]] <- unique(treatelements[[1]]$data$id)
    }
  }
  missingIds  <- c()
  if (length(ids) > 1) {
    missingIds <- setdiff(Reduce(union, unname(ids)), Reduce(intersect, unname(ids)))
  }
  if (length(missingIds) > 0) {
    # BUG in Simulx 2020R1 & 2021R1
    warning("Due to a bug in Simulx, the ids ", paste(missingIds, collapse = ", "),
            " will be excluded from the simulation because they do not appear in some of the treatment and output tables.",
            call. = FALSE)
  }
}

# CHeck functions --------------------------------------------------------------
.check_bool <- function(bool, argname) {
  if (!is.logical(bool))
    stop("Invalid ", argname, ". It must be logical.", call. = F)
  return(bool)
}

.check_string <- function(str, argname) {
  if (!is.string(str))
    stop("Invalid ", argname, ". It must be a string.", call. = F)
  return(str)
}

.check_strict_pos_integer <- function(int, argname) {
  if(!(is.double(int)||is.integer(int)))
    stop("Invalid ", argname, ". It must be a strictly positive integer.", call. = F)
  if ((int <= 0) || (!as.integer(int) == int))
    stop("Invalid ", argname, ". It must be a strictly positive integer.", call. = F)
  return(int)
}

.check_in_range <- function(arg, argname, lowerbound = -Inf, upperbound = Inf, includeBound = TRUE) {
  if (includeBound) {
    if (! all(arg >= lowerbound) | ! all(arg <= upperbound)) {
      stop(argname, " must be in [", lowerbound, ", ", upperbound, "].", call. = F)
    }
  } else {
    if (! all(arg > lowerbound) | ! all(arg < upperbound)) {
      stop(argname, " must be in ]", lowerbound, ", ", upperbound, "[.", call. = F)
    }
  }
  return(invisible(TRUE))
}

is.string <- function(str) {
  isStr <- TRUE
  if (!is.null(names(str))) {
    isStr <- FALSE
  } else if (length(str) > 1) {
    isStr <- FALSE
  } else if (! is.character(str)) {
    isStr <- FALSE
  }
  return(isStr)
}

.check_in_vector <- function(arg, argname, vector) {
  if (is.null(arg)) return(invisible(TRUE))
  if (! all(is.element(arg, vector))) {
    stop(argname, " must be in {'", paste(vector, collapse="', '"), "'}.", call. = F)
  }
  return(invisible(TRUE))
}

.check_double <- function(d, argname) {
  if(!(is.double(d)||is.integer(d)))
    stop("Invalid ", argname, ". It must be a double.", call. = F)
  return(d)
}
