#' Strategy classification using the legacy Garthe classifier.
#'
#' Calculates strategies using a method based on Garthe et. al. 2009.
#'
#' @param metrics An \code{rtrack_metrics} object from
#'   \code{\link{calculate_metrics}} or a list of such objects.
#' @param parameters A \code{\link[base]{data.frame}} of parameters to adjust
#'   output. Currently not implemented.
#'
#' @return An \code{rtrack_strategies} object. The \code{calls} element contains
#'   the called strategy/strategies as well as several additional metrics
#'   generated by this method.
#'
#' @seealso \code{\link{call_strategy}}.
#'
#' @examples
#' require(Rtrack)
#' track_file <- system.file("extdata", "Track_1.csv", package = "Rtrack")
#' arena_description <- system.file("extdata", "Arena_SW.txt", package = "Rtrack")
#' arena <- read_arena(arena_description)
#' path <- read_path(track_file, arena, track.format = "ethovision.3.csv")
#' metrics <- calculate_metrics(path, arena)
#' strategies <- call_mwm_strategy_garthe(metrics)
#' # Inspect the strategy call
#' strategies$calls
#'
#' @importFrom stats na.omit
#'
#' @export
call_mwm_strategy_garthe = function(metrics, parameters = NULL) {
	if(!(is(metrics, "rtrack_metrics")) & !(is(metrics, "list") & is(metrics[[1]], "rtrack_metrics"))){
		stop("This function requires a 'rtrack_metrics' object or a list of 'rtrack_metrics' objects. Did you create this with 'calculate_metrics' or 'read.mwm.experiment'?")
	}else{
		strategy.names = c(
			"1" = "thigmotaxis",
			"2" = "random swimming",
			"3" = "scanning",
			"4" = "chaining",
			"5" = "directed search",
			"6" = "focal search",
			"7" = "direct swimming",
			"8" = "perseverance",
			"0" = "unknown"
		)
		strategy.colours = c(
			"1" = "#E32423", 
			"2" = "#F28B1D", 
			"3" = "#F4E300", 
			"4" = "#C2D01E", 
			"5" = "#67AE25", 
			"6" = "#6EC5D9", 
			"7" = "#344A97",
			"8" = "black", 
			"0" = "white"
		)
		metrics.list = metrics # By default assume a list has been passed...
		if((is(metrics, "rtrack_metrics"))){
			# ...if not, wrap the single 'rtrack_metrics' object into a list of length 1 and run the lapply below
			metrics.list = list(metrics)
		}
		calls = t(sapply(metrics.list, function(metrics){
			## Definitions of the following parameters translated from Matlab:
			## meanalpha
			# alpha(j,1)=atand((m2-m1)/(1+m1*m2));
			# where, for each point, m1 is the slope between current to next position
			# and m2 is the slope from current position to goal
			# atand is the inverse tangent in degrees
			# Thus alpha is the angle (in degrees) between the _next_ path section and the current heading to goal
			meanalpha = mean(metrics$alpha, na.rm = TRUE)
			# 
			## eff
			# The efficiency is simply abs(alpha(j,1))<15
			# i.e. the number of times the absolute anpha exceeded 15 degrees
			# Then converted into a percentage of all points
			eff = sum(metrics$alpha > 15, na.rm = TRUE) / length(stats::na.omit(metrics$alpha)) * 100
			#
			## dtPOG, dtGOAL, dtOLDGOAL
			# These metrics use a normalised distance;
			# dtPOGim=sqrt((xpos3(j,1)-POGX(1,trial))^2+(ypos3(j,1)-POGY(1,trial))^2);
			# dtPOGsum=dtPOGsum+dtPOGim;
			# dtPOG(1,trial)=dtPOGsum/rows_real;
			# The variable xpos3 = xpos2
			# xpos2(j,trial)=((xpos(j,trial)-xmin)*1/(xmax-xmin));
			# So this is just the x coordinate as a fraction of the diameter 
			# (e.g. 0 at the bottom and 1 at the top)
			# This should be equivalent to normalising the mean absolute distances
			dtPOG = unname(metrics$summary["mean.d.origin"] / (metrics$arena$pool$radius * 2))
			dtGOAL = unname(metrics$summary["mean.d.goal"] / (metrics$arena$pool$radius * 2))
			dtOLDGOAL = unname(metrics$summary["mean.d.old.goal"] / (metrics$arena$pool$radius * 2))
			# If there is no old goal, then we can just make dtOLDGOAL equal to the maximum
			if(is.na(dtOLDGOAL)) dtOLDGOAL = 1
			# 
			## dtCENTER
			# This is a bit different, as it uses xpos2b and ypos2b
			# xpos2b(j,trial)=((xpos(j,trial)-xmin)*2/(xmax-xmin))-1;
			# This is normalised and scaled around zero
			# This should not alter the relative distances though
			dtCENTER = unname(metrics$summary["mean.d.origin"] / (metrics$arena$pool$radius * 2))
			## annuluszonerel, wallzonerel
			# These are measured as the number of path points in the zone
			# and normalised to a fraction of path length
			# This is equivalent to normalised 'time in zone' measures (already as fraction of total swim path)
			annuluszonerel = unname(metrics$summary["time.in.zone.annulus"])
			wallzonerel = unname(metrics$summary["time.in.zone.wall"])
			#
			## covsurfacerel
			# The 'surface covered' is simply an ellipse minimally containing all points in the path
			# i.e. pi * (max(x) - min(x)) * (max(y) - min(y))
			# and then normalised (?)
			# covsurface(1,trial)=pi*(xdist(1,trial)*ydist(1,trial));
			# covsurfacemax=pi*1*1;
			# covsurfacerel=(covsurface/covsurfacemax);
			# I'm not sure this was the intended result. i.e. covsurfacerel = (x_diameter) * (y_diameter)
			# Maybe I am misunderstanding the Matlab code here
			# Surely this should be based on radii and normalised to the total arena/pool area
			covsurfacerel = unname((pi * ((max(metrics$path$x) - min(metrics$path$x)) / 2) * ((max(metrics$path$y) - min(metrics$path$y)) / 2)) / metrics$area["pool"])
			#
			## outliersrel
			# x1=xpos2_temp(1,1);
			# y1=ypos2_temp(1,1);
			# x2=GOALX;
			# y2=GOALY;
			# x3=xpos2_temp(j-1,1);
			# y3=ypos2_temp(j-1,1);
			# x4=xpos2_temp(j,1);
			# y4=ypos2_temp(j,1);
			# ang=atand((y2-y1)/(x2-x1))-atand((y4-y3)/(x4-x3));
			# if abs(ang)>22.5
			# outliers=outliers+1
			# So the angle 'ang' is the difference between the direct path from start to goal
			# and the _previous_ path section (c.f. 'alpha')
			# Thus, it is similar to the initial heading error (but calculated over all points)
			outliersrel = sum(metrics$heading.error > 22.5, na.rm = TRUE) / length(stats::na.omit(metrics$heading.error))
			#
			strategy = NA
			if(meanalpha < 25  & eff > 80){
			    strategy = "7";
			}else if(dtPOG < 0.25 & dtGOAL < 0.18){
			    strategy = "6";
			}else if(dtOLDGOAL < 0.25 & dtPOG < 0.2 & dtGOAL > 0.3){
			    strategy = "8";
			}else if(outliersrel < 0.45 & dtGOAL < 0.28){
			    strategy = "5";
			}else if(annuluszonerel > 0.5 & dtCENTER > 0.65){
			    strategy = "4";
			}else if(covsurfacerel < 0.7 & wallzonerel < 0.75 & dtCENTER < 0.6){
			    strategy = "3";
			}else if(wallzonerel > 0.65 & dtCENTER > 0.58){
			    strategy = "1";
			}else if(covsurfacerel > 0.35 & wallzonerel < 0.8){
			    strategy = "2";
			}else{
				strategy = "0";
			}
			as.data.frame(cbind("strategy" = strategy, "name" = paste0("", strategy.names[strategy]), "meanalpha" = round(meanalpha, 2), "eff" = round(eff, 2), "dtPOG" = round(dtPOG, 2), "dtGOAL" = round(dtGOAL, 2), "dtOLDGOAL" = round(dtOLDGOAL, 2), "dtCENTER" = round(dtCENTER, 2), "annuluszonerel" = round(annuluszonerel, 2), "wallzonerel" = round(wallzonerel, 2), "covsurfacerel" = round(covsurfacerel, 2), "outliersrel" = round(outliersrel, 2)), stringsAsFactors = FALSE)
		}))
	
		strategies = list(
			method = "garthe",
			version = "1",
			parameters = NULL,
			strategy.names = strategy.names,
			strategy.colours = strategy.colours,
			plot.order = c(9, 7:1, 8),
			calls = as.data.frame(calls, stringsAsFactors = FALSE),
			thresholded = FALSE
		)
		class(strategies) = "rtrack_strategies"
		return(strategies)
	}
}
