\name{modelSim.dynamicsSVM}
\alias{modelSim.dynamicsSVM}
\alias{modelSim}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Simulation from Stochastic Volatility Models with Jumps
}
\description{
The \code{modelSim} function generates returns and variances for a wide class of stochastic volatility models.
}
\usage{
\method{modelSim}{dynamicsSVM}(dynamics, t, init_vol = 0.032, ...)
}
\arguments{
 \item{dynamics}{A dynamicsSVM object representing the model dynamics to be used for simulating data.}
 \item{t}{Number of observations to be simulated.}
 \item{init_vol}{Initial value of the volatility factor (e.i., value of \code{x_0}).}
  \item{...}{Further arguments passed to or from other methods.}
}
\value{
\item{volatility_factor}{Vector of the instantaneous volatility factor values generated by the \code{modelSim} function.}
\item{returns}{Vector of the returns generated by the \code{modelSim} function.}
}
\examples{
set.seed(1)
# Generate 250 returns from the DuffiePanSingleton model
DuffiePanSingleton_mod <- dynamicsSVM(model = "DuffiePanSingleton") 
DuffiePanSingleton_sim <- modelSim(t = 200, dynamics = DuffiePanSingleton_mod) 

# Plot the volatility factor and returns that were generated
plot(DuffiePanSingleton_sim$volatility_factor, type = 'l',
  main = 'DuffiePanSingleton Model Simulated Volatility Factor', ylab = 'Volatility Factor')
 
plot(DuffiePanSingleton_sim$returns, type = 'l',
  main = 'DuffiePanSingleton Model Simulated Returns', ylab = 'Returns')
 
# Generate 250 steps from a custom model
# Set parameters
kappa <- 100; theta <- 0.05; sigma <- 2.3; h <- 1/252 ; mu <- 0.04
rho <- -0.8; omega <- 5; alpha <- -0.025; nu <- 0.01; rho_z <- -1; delta <- 0.025
# Jump compensator
alpha_bar <- exp(alpha + 0.5 * delta^2) / (1 - rho_z * nu) - 1

# Define returns drift and diffusion functions
# Returns drift and diffusion
mu_y <- function(x, mu, alpha_bar, omega, h){
  return(h * (mu - x / 2 - alpha_bar * omega))
}
mu_y_params <- list(mu, alpha_bar, omega , h)
sigma_y <- function(x, h, sigma){
  return(sigma * sqrt(h) * pmax(x,0))
}
sigma_y_params <- list(h, sigma)

# Volatility factor drift and diffusion functions
mu_x <- function(x, h, kappa, theta){
  return(x + h * kappa * pmax(0,x) * (theta - pmax(0,x)))
}
mu_x_params <- list(h, kappa, theta)

sigma_x <- function(x, sigma, h){
  return(sigma * sqrt(h) * pmax(0,x))
}
sigma_x_params <- list(sigma, h)

# Include simultaneous return and volatility factor jumps
# based on the Poisson distribution for jump times
jump_dist <- rpois
jump_params <- list(omega * h)
custom_mod <- dynamicsSVM(model = "Custom", mu_x = mu_x, mu_y = mu_y,
  sigma_x = sigma_x, sigma_y = sigma_y,
  mu_x_params = mu_x_params, mu_y_params = mu_y_params,
  sigma_x_params = sigma_x_params, sigma_y_params = sigma_y_params,
  jump_dist = jump_dist, jump_params = jump_params, 
  nu = nu, rho_z = rho_z, omega = omega, alpha = alpha, delta = delta)
custom <- modelSim(t = 250, dynamics = custom_mod)
  
plot(custom$volatility_factor, type = 'l',
  main = 'Custom Model Simulated Volatility Factor', ylab = 'Volatility Factor')
plot(custom$returns, type = 'l',
  main = 'Custom Model Simulated Returns', ylab = 'Returns')
}
