% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ABCRef.R
\name{ABCRef}
\alias{ABCRef}
\title{Produces ABC reference table}
\usage{
ABCRef(npart, priors, pars, func, sumNames, parallel = FALSE,
  mc.cores = NA, ...)
}
\arguments{
\item{npart}{The number of particles (must be a positive integer).}

\item{priors}{A \code{data.frame} containing columns \code{parnames}, \code{dist}, \code{p1} and 
\code{p2}, with number of rows equal to the number of parameters. The column
\code{parname} simply gives names to each parameter for plotting and summarising.
Each entry in the \code{dist} column must contain one of \code{c("unif", "norm", "gamma")}, 
and the corresponding \code{p1} and \code{p2} entries relate to the hyperparameters 
(lower and upper bounds in the uniform case; mean and standard deviation in the 
normal case; and shape and rate in the gamma case).}

\item{pars}{A named vector or matrix of parameters to use for the simulations. If \code{pars} is a vector then
this is repeated `npart` times, else it must be a \code{matrix} with `npart` rows. 
You cannot specify both `pars` and `priors`.}

\item{func}{Function that runs the simulator. The first argument must be \code{pars}. The function
must return a \code{vector} of simulated summary measures, or a missing value (\code{NA})
if there is an error. The output from the function must be a vector with length equal 
to \code{length(sumNames)}.}

\item{sumNames}{A \code{character} vector of summary statistic names.}

\item{parallel}{A \code{logical} determining whether to use parallel processing or not.}

\item{mc.cores}{Number of cores to use if using parallel processing.}

\item{...}{Extra arguments to be passed to \code{func}.}
}
\value{
An \code{data.frame} object with \code{npart} rows, where the first \code{p} columns correspond to 
        the proposed parameters, and the remaining columns correspond to the simulated outputs.
}
\description{
Produces reference table of simulated outcomes for use in various
    Approximate Bayesian Computation (ABC) algorithms.
}
\details{
Runs simulations for a large number of particles, either pre-specified or
    sampled from the a set of given prior distributions. Returns a table of summary 
    statistics for each particle. Useful for deciding on initial tolerances during an
    \code{\link{ABCSMC}} run, or for producing a reference table to use in e.g. the 
    ABC with Random Forests approach of Raynal et al. (2017).
}
\examples{
\donttest{
## set up SIR simulation model
transitions <- c(
    "S -> beta * S * I -> I", 
    "I -> gamma * I -> R"
)
compartments <- c("S", "I", "R")
pars <- c("beta", "gamma")
model <- mparseRcpp(
    transitions = transitions, 
    compartments = compartments,
    pars = pars
)
model <- compileRcpp(model)

## generate function to run simulators
## and produce final epidemic size and time
## summary statistics
simRef <- function(pars, model) {
    ## run model over a 100 day period with
    ## one initial infective in a population
    ## of 120 individuals
    sims <- model(pars, 0, 100, c(119, 1, 0))
    
    ## return vector of summary statistics
    c(finaltime = sims[2], finalsize = sims[5])
}

## set priors
priors <- data.frame(
    parnames = c("beta", "gamma"), 
    dist = rep("gamma", 2), 
    stringsAsFactors = FALSE
)
priors$p1 <- c(10, 10)
priors$p2 <- c(10^4, 10^2)

## produce reference table by sampling from priors
## (add additional arguments to 'func' at the end)
refTable <- ABCRef(
    npart = 100, 
    priors = priors, 
    func = simRef, 
    sumNames = c("finaltime", "finalsize"),
    model = model
)
refTable
}

}
\references{
Raynal, L, Marin J-M, Pudlo P, Ribatet M, Robert CP and Estoup A. (2017) <ArXiv:1605.05537>
}
