## SimInf, a framework for stochastic disease spread simulations
## Copyright (C) 2015 - 2016  Stefan Engblom
## Copyright (C) 2015 - 2016  Stefan Widgren
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

library(SimInf)

## For debugging
sessionInfo()

## Check invalid u0
res <- tools::assertError(SISe3_sp(u0 = "u0"))
stopifnot(length(grep("'u0' must be a data.frame",
                      res[[1]]$message)) > 0)

u0 <- structure(list(S_1 = c(0, 1, 2, 3, 4, 5, 6, 7, 8),
                       I_1 = c(0, 0, 0, 0, 0, 0, 0, 0, 0),
                       S_2 = c(0, 1, 2, 3, 4, 5, 6, 7, 8),
                       I_2 = c(0, 0, 0, 0, 0, 0, 0, 0, 0),
                       S_3 = c(0, 1, 2, 3, 4, 5, 6, 7, 8),
                       I_3 = c(0, 0, 0, 0, 0, 0, 0, 0, 0)),
                  .Names = c("S_1", "I_1", "S_2", "I_2", "S_3", "I_3"),
                  row.names = c(NA, -9L), class = "data.frame")

## Place nodes in a grid
distance <- expand.grid(x = seq_len(3),
                        y = seq_len(3))
distance <- distance_matrix(distance$x, distance$y, 2)

## Check missing columns in u0
res <- tools::assertError(
    SISe3_sp(u0 = u0[, c("I_1", "S_2", "I_2", "S_3", "I_3")]))
stopifnot(length(grep("Missing columns in u0",
                      res[[1]]$message)) > 0)

res <- tools::assertError(
    SISe3_sp(u0 = u0[, c("S_1", "S_2", "I_2", "S_3", "I_3")]))
stopifnot(length(grep("Missing columns in u0",
                      res[[1]]$message)) > 0)

res <- tools::assertError(
    SISe3_sp(u0 = u0[, c("S_1", "I_1", "I_2", "S_3", "I_3")]))
stopifnot(length(grep("Missing columns in u0",
                      res[[1]]$message)) > 0)

res <- tools::assertError(
    SISe3_sp(u0 = u0[, c("S_1", "I_1", "S_2", "S_3", "I_3")]))
stopifnot(length(grep("Missing columns in u0",
                      res[[1]]$message)) > 0)

res <- tools::assertError(
    SISe3_sp(u0 = u0[, c("S_1", "I_1", "S_2", "I_2", "I_3")]))
stopifnot(length(grep("Missing columns in u0",
                      res[[1]]$message)) > 0)

res <- tools::assertError(
    SISe3_sp(u0 = u0[, c("S_1", "I_1", "S_2", "I_2", "S_3")]))
stopifnot(length(grep("Missing columns in u0",
                      res[[1]]$message)) > 0)

## Check missing upsilon_1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_1' is missing",
                      res[[1]]$message)) > 0)

## Check missing upsilon_2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_2' is missing",
                      res[[1]]$message)) > 0)

## Check missing upsilon_3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_3' is missing",
                      res[[1]]$message)) > 0)

## Check missing gamma_1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_1' is missing",
                      res[[1]]$message)) > 0)

## Check missing gamma_2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_2' is missing",
                      res[[1]]$message)) > 0)

## Check missing gamma_3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_3' is missing",
                      res[[1]]$message)) > 0)

## Check missing alpha
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'alpha' is missing",
                      res[[1]]$message)) > 0)

## Check missing beta_t1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t1' is missing",
                      res[[1]]$message)) > 0)

## Check missing beta_t2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t2' is missing",
                      res[[1]]$message)) > 0)

## Check missing beta_t3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t3' is missing",
                      res[[1]]$message)) > 0)

## Check missing beta_t4
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t4' is missing",
                      res[[1]]$message)) > 0)

## Check missing end_t1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t1' is missing",
                      res[[1]]$message)) > 0)

## Check missing end_t2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t2' is missing",
                      res[[1]]$message)) > 0)

## Check missing end_t3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t3' is missing",
                      res[[1]]$message)) > 0)

## Check missing end_t4
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t4' is missing",
                      res[[1]]$message)) > 0)

## Check missing coupling
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   distance  = distance))
stopifnot(length(grep("'coupling' is missing",
                      res[[1]]$message)) > 0)

## Check missing distance
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005))
stopifnot(length(grep("'distance' is missing",
                      res[[1]]$message)) > 0)

## Check non-numeric upsilon_1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = "0.0357",
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_1' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric upsilon_2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = "0.0357",
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_2' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric upsilon_3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = "0.00935",
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_3' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric gamma_1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = "0.1",
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_1' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric gamma_2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = "0.1",
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_2' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric gamma_3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = "0.1",
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_3' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric alpha
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = "1.0",
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'alpha' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric beta_t1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = "0.19",
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t1' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric beta_t2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = "0.085",
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t2' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric beta_t3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = "0.075",
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t3' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-numeric beta_t4
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = "0.185",
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t4' must be numeric",
                      res[[1]]$message)) > 0)

## Check non-integer end_t1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = "91",
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t1' must be integer",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91.5,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t1' must be integer",
                      res[[1]]$message)) > 0)

## Check non-integer end_t2
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = "182",
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t2' must be integer",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182.5,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t2' must be integer",
                      res[[1]]$message)) > 0)

## Check non-integer end_t3
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = "273",
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t3' must be integer",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273.5,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t3' must be integer",
                      res[[1]]$message)) > 0)

## Check non-integer end_t4
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = "365",
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t4' must be integer",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365.5,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t4' must be integer",
                      res[[1]]$message)) > 0)

## Check non-numeric coupling
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = "0.0005",
                                   distance  = distance))
stopifnot(length(grep("'coupling' must be numeric",
                      res[[1]]$message)) > 0)

## Check that length of upsilon_1 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = c(0.0357, 0.0357),
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_1' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of upsilon_2 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = c(0.0357, 0.0357),
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_2' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of upsilon_3 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = c(0.00935, 0.00935),
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'upsilon_3' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of gamma_1 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = c(0.1, 0.1),
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_1' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of gamma_2 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = c(0.1, 0.1),
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_2' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of gamma_3 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = c(0.1, 0.1),
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'gamma_3' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of alpha equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = c(1.0, 1.0),
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'alpha' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of beta_t1 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = c(0.19, 0.19),
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t1' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of beta_t2 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = c(0.085, 0.085),
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t2' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of beta_t3 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = c(0.075, 0.075),
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t3' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of beta_t4 equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = c(0.185, 0.185),
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'beta_t4' must be of length 1",
                      res[[1]]$message)) > 0)

## Check that length of end_t1 equals 1 or nrow(u0)
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = c(91, 91),
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t1' must be of length 1 or 'nrow\\(u0\\)'",
                      res[[1]]$message)) > 0)

## Check that length of end_t2 equals 1 or nrow(u0)
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = c(182, 182),
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t2' must be of length 1 or 'nrow\\(u0\\)'",
                      res[[1]]$message)) > 0)

## Check that length of end_t3 equals 1 or nrow(u0)
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = c(273, 273),
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t3' must be of length 1 or 'nrow\\(u0\\)'",
                      res[[1]]$message)) > 0)

## Check that length of end_t4 equals 1 or nrow(u0)
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = c(365, 365),
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t4' must be of length 1 or 'nrow\\(u0\\)'",
                      res[[1]]$message)) > 0)

## Check that length of coupling equals 1
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = c(0.0005, 0.0005),
                                   distance  = distance))
stopifnot(length(grep("'coupling' must be of length 1",
                      res[[1]]$message)) > 0)

## Check interval endpoints
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = -1,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t1' must be greater than or equal to '0'",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 18,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t1' must be less than 'end_t2'",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 173,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t2' must be less than 'end_t3'",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 365,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t3' must be less than '364'",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = -1,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t4' must be greater than or equal to '0'",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 366,
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep("'end_t4' must be less than or equal to '365'",
                      res[[1]]$message)) > 0)

res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 4:12,
                                   end_t2    = 5:13,
                                   end_t3    = c(8:15, 19),
                                   end_t4    = c(2, 11:18),
                                   coupling  = 0.0005,
                                   distance  = distance))
stopifnot(length(grep(
    "'end_t4' must be less than 'end_t1' or greater than 'end_t3'",
    res[[1]]$message)) > 0)

## Check distance matrix
res <- tools::assertError(SISe3_sp(u0        = u0,
                                   tspan     = seq_len(10) - 1,
                                   events    = NULL,
                                   phi       = rep(1, nrow(u0)),
                                   upsilon_1 = 0.0357,
                                   upsilon_2 = 0.0357,
                                   upsilon_3 = 0.00935,
                                   gamma_1   = 0.1,
                                   gamma_2   = 0.1,
                                   gamma_3   = 0.1,
                                   alpha     = 1.0,
                                   beta_t1   = 0.19,
                                   beta_t2   = 0.085,
                                   beta_t3   = 0.075,
                                   beta_t4   = 0.185,
                                   end_t1    = 91,
                                   end_t2    = 182,
                                   end_t3    = 273,
                                   end_t4    = 365,
                                   coupling  = 0.0005,
                                   distance  = as.matrix(distance)))
stopifnot(length(grep("The 'distance' argument must be of type 'dgCMatrix'",
                      res[[1]]$message)) > 0)

## Check 'suscpetible' and 'infected' methods
model <- SISe3_sp(u0        = u0,
                  tspan     = seq_len(10) - 1,
                  events    = NULL,
                  phi       = rep(0, nrow(u0)),
                  upsilon_1 = 0.0357,
                  upsilon_2 = 0.0357,
                  upsilon_3 = 0.00935,
                  gamma_1   = 0.1,
                  gamma_2   = 0.1,
                  gamma_3   = 0.1,
                  alpha     = 1.0,
                  beta_t1   = 0.19,
                  beta_t2   = 0.085,
                  beta_t3   = 0.075,
                  beta_t4   = 0.185,
                  end_t1    = 91,
                  end_t2    = 182,
                  end_t3    = 273,
                  end_t4    = 365,
                  coupling  = 0.0005,
                  distance  = distance)

result <- run(model, threads = 1)

S_expected <- structure(c(0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L,
                          0L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L),
                        .Dim = 9:10)

S_observed <- susceptible(result, age = 1)

stopifnot(identical(S_observed, S_expected))

I_expected <- structure(c(0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L,
                          0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L),
                        .Dim = 9:10)

I_observed <- infected(result, age = 1)

stopifnot(identical(I_observed, I_expected))

## Check SISe3_sp plot method
pdf_file <- tempfile(fileext = ".pdf")
pdf(pdf_file)
plot(result, t0 = "2015-01-01")
dev.off()
stopifnot(file.exists(pdf_file))
unlink(pdf_file)

## Check that C SISe3_sp run function fails for misspecified SISe3_sp model
res <- .Call("SISe3_sp_run", NULL, NULL, NULL, PACKAGE = "SimInf")
stopifnot(identical(res$error, -10L))

res <- .Call("SISe3_sp_run", "SISe3_sp", NULL, NULL, PACKAGE = "SimInf")
stopifnot(identical(res$error, -10L))

## Check error non-finite v
model <- SISe3_sp(u0        = u0,
                  tspan     = seq_len(10) - 1,
                  events    = NULL,
                  phi       = rep(1, nrow(u0)),
                  upsilon_1 = 0.0357,
                  upsilon_2 = 0.0357,
                  upsilon_3 = 0.00935,
                  gamma_1   = 0.1,
                  gamma_2   = 0.1,
                  gamma_3   = 0.1,
                  alpha     = 1.0,
                  beta_t1   = 0.19,
                  beta_t2   = 0.085,
                  beta_t3   = 0.075,
                  beta_t4   = 0.185,
                  end_t1    = 91,
                  end_t2    = 182,
                  end_t3    = 273,
                  end_t4    = 365,
                  coupling  = 0.0005,
                  distance  = distance)
model@gdata <- rep(Inf, length(model@gdata))
res <- tools::assertError(run(model, threads = 1))
stopifnot(length(grep("The continuous state 'v' is not finite.",
                      res[[1]]$message)) > 0)

## Check negative v
u0 <- structure(list(S_1 = c(0, 1, 2, 3, 4, 5, 6, 7, 8),
                       I_10 = c(10, 10, 10, 10, 10, 10, 10, 10, 10),
                       S_2 = c(0, 1, 2, 3, 4, 5, 6, 7, 8),
                       I_2 = c(10, 10, 10, 10, 10, 10, 10, 10, 10),
                       S_3 = c(0, 1, 2, 3, 4, 5, 6, 7, 8),
                       I_3 = c(10, 10, 10, 10, 10, 10, 10, 10, 10)),
                  .Names = c("S_1", "I_1", "S_2", "I_2", "S_3", "I_3"),
                  row.names = c(NA, -9L), class = "data.frame")
model <- SISe3_sp(u0        = u0,
                  tspan     = seq_len(10) - 1,
                  events    = NULL,
                  phi       = rep(1, nrow(u0)),
                  upsilon_1 = 0,
                  upsilon_2 = 0,
                  upsilon_3 = 0,
                  gamma_1   = 0,
                  gamma_2   = 0,
                  gamma_3   = 0,
                  alpha     = -1.0,
                  beta_t1   = 0.19,
                  beta_t2   = 0.085,
                  beta_t3   = 0.075,
                  beta_t4   = 0.185,
                  end_t1    = 91,
                  end_t2    = 182,
                  end_t3    = 273,
                  end_t4    = 365,
                  coupling  = 0.0005,
                  distance  = distance)
res <- tools::assertError(run(model, threads = 1))
stopifnot(length(grep("The continuous state 'v' is negative.",
                      res[[1]]$message)) > 0)
