% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/formatDT.R
\name{formatDT}
\alias{formatDT}
\title{Converts date or datetime strings into alternate formats}
\usage{
formatDT(dt, date.outformat = NULL, time.outformat = NULL, posix = TRUE,
  weekday = FALSE)
}
\arguments{
\item{dt}{A character vector of date values or datetime values}

\item{date.outformat}{A character string requesting the date format to be
returned.  The following date outformats are supported: "mm/dd/yyyy",
"mm-dd-yyyy", "yyyy-mm-dd", "yyyymmdd", "ddmonyyyy", and "dd-mon-yyyy".  If
\code{date.outformat = NULL}, then "mm/dd/yyyy" is used.}

\item{time.outformat}{A character string requesting the time format to be
returned.  The following time outformats are supported: "hh:mm:sspm",
"hh:mm:ss pm", "hh:mm:ss", "hh:mmpm", "hh:mm pm", and "hh:mm".  If
\code{time.outformat = NULL}, then "hh:mm:ss pm" is used.}

\item{posix}{\code{= TRUE} returns date and datetime vectors of class
POSIXct that can be used for time calculations.}

\item{weekday}{\code{= TRUE} returns a character vector denoting the day of
the week.}
}
\value{
A list with these components: \item{date}{A character vector of the
form requested by \code{date.outformat}.} \item{time}{A character vector of
the form requested by \code{time.outformat} or an empty character vector of
the form "" if the time is not present in the input vector \code{dt}.}
\item{dt}{A character vector containing the combined datetime using the
requested formats.  If time is not present in the input vector \code{dt},
then simply the date is returned.} \item{date.posix}{A vector of class
"POSIXt POSIXct" containing the date.  This is only returned if
\code{posix = TRUE}.} \item{dt.posix}{A vector of class "POSIXt POSIXct"
containing the datetime.  This is only returned if \code{posix = TRUE} and
time values are present in the argument \code{dt}.} \item{weekday}{A
character vector indicating the days of the week.  This is only returned if
\code{weekday = TRUE}.}
}
\description{
Can be used to convert date-time character vectors into other types of
date-time formats.  It is designed to automatically find the appropriate
date and time informats without the user having to specify them.
}
\details{
If the input vector contains times, \code{formatDT} assumes that the dates
and times are separated by at least one space.  The date format and the
time format of the input vector must be the same for all cells in the
vector. The input format is determined by the first non-missing entry of
the \code{dt} vector. Missing values (\code{NA} or \code{""}) are carried
through to the output vectors without error.

In chosing the informat, \code{formatDT} first checks if the datetime
string has a format of "dd/mm/yyyy hh:mm:ss pm".  If so, it moves directly
to the datetime conversions.  Otherwise, it searches through the date and
time informats listed below for a suitable match.

Acceptable date informats for \code{dt}: \code{mm/dd/yyyy},
\code{mm-dd-yyyy}, \code{yyyy-mm-dd}, \code{yyyymmdd}, \code{ddmonyyyy},
\code{dd-mon-yyyy}

Acceptable time informats for \code{dt}: \code{hh:mm:sspm}, \code{hh:mm:ss
pm}, \code{hh:mm:ss} (24 hour time), \code{hh:mmpm}, \code{hh:mm pm},
\code{hh:mm} (24 hour time), \code{hhmm} (24 hour time), \code{hhmmss} (24
hour time)
}
\examples{
# Demonstrates conversion of different datetime informats
formatDT("03/12/2004 04:31:17pm", posix = FALSE)
formatDT("12Mar2004 04:31pm", posix = FALSE)
formatDT("2004-3-12 16:31:17", posix = FALSE)
formatDT("7-5-1998 22:13")

# Specifying different types of outformats
formatDT("03/12/2004", date.outformat = "dd-mon-yyyy", posix = FALSE)
formatDT("17-Sep-1782 12:31am", date.outformat = "yyyy-mm-dd",
         time.outformat = "hh:mm", posix = FALSE)

# Processing datetime vectors
formatDT(c("03/12/2004 04:31pm","03/12/2005 04:32:18pm"), posix = FALSE)
formatDT(c("03/12/2004 04:31:17pm","03/12/2005 04:32:18pm"))
formatDT(c("03/12/2004 04:31:17pm","03/12/2005 04:32:18pm"), weekday = TRUE)

# An incorrect date (will produce an error)
try(formatDT("29-Feb-2001"))

# An incorrect time will also produce an error
try(formatDT("28-Feb-2001 00:00:00 AM"))
formatDT("28-Feb-2001 12:00:00 AM")

# Illustrate the handling of missing values
formatDT(c(NA,"","2010-10-23 3:47PM"), weekday = TRUE)
}
\author{
Landon Sego
}
\keyword{misc}

