% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation-spades.R
\docType{methods}
\name{spades}
\alias{spades}
\alias{spades,simList,ANY,ANY,missing-method}
\alias{spades,ANY,ANY,ANY,logical-method}
\title{Run a spatial discrete event simulation}
\usage{
spades(sim, debug = getOption("spades.debug"), progress = NA, cache,
  .plotInitialTime = NULL, .saveInitialTime = NULL, notOlderThan = NULL,
  ...)

\S4method{spades}{simList,ANY,ANY,missing}(sim,
  debug = getOption("spades.debug"), progress = NA, cache,
  .plotInitialTime = NULL, .saveInitialTime = NULL, notOlderThan = NULL,
  ...)

\S4method{spades}{ANY,ANY,ANY,logical}(sim, debug = getOption("spades.debug"),
  progress = NA, cache, .plotInitialTime = NULL, .saveInitialTime = NULL,
  notOlderThan = NULL, ...)
}
\arguments{
\item{sim}{A \code{simList} simulation object, generally produced by \code{simInit}.}

\item{debug}{Optional logical flag or character vector indicating what to print to
console at each event. See details.
Default is to use the value in \code{getOption("spades.debug")}.}

\item{progress}{Logical (\code{TRUE} or \code{FALSE} show a graphical progress bar),
character (\code{"graphical"}, \code{"text"}) or numeric indicating
the number of update intervals to show in a graphical progress bar.}

\item{cache}{Logical. If \code{TRUE}, then the \code{spades} call will be cached.
This means that if the call is made again with the same simList,
then `spades`` will return the return value from the previous run
of that exact same simList. Default \code{FALSE}. See Details.
See also the vignette on caching for examples.}

\item{.plotInitialTime}{Numeric. Temporarily override the \code{.plotInitialTime}
parameter for all modules. See Details.}

\item{.saveInitialTime}{Numeric. Temporarily override the \code{.plotInitialTime}
parameter for all modules. See Details.}

\item{notOlderThan}{Date or time. Passed to \code{reproducible::Cache} to update the cache.
Default is \code{NULL}, meaning don't update the cache.
If \code{Sys.time()} is provided, then it will force a recache,
i.e., remove old value and replace with new value.
Ignored if \code{cache} is \code{FALSE}.}

\item{...}{Any. Can be used to make a unique cache identity, such as "replicate = 1".
This will be included in the \code{Cache} call, so will be unique
and thus \code{spades} will not use a cached copy as long as
anything passed in \code{...} is unique, i.e., not cached previously.}
}
\value{
Invisibly returns the modified \code{simList} object.
}
\description{
Here, we implement a simulation in a more modular fashion so it's easier to add
submodules to the simulation. We use S4 classes and methods, and use `data.table`
instead of `data.frame` to implement the event queue (because it is much faster).
}
\details{
The is the workhorse function in the SpaDES package. It runs simulations by
implementing the rules outlined in the \code{simList}.

This function gives simple access to two sets of module parameters:
\code{.plotInitialTime} and with \code{.plotInitialTime}. The primary use of
these arguments is to temporarily turn off plotting and saving. "Temporary"
means that the \code{simList} is not changed, so it can be used again with
the simList values reinstated. To turn off plotting and saving, use
\code{.plotInitialTime = NA} or \code{.saveInitialTime = NA}. NOTE: if a
module did not use \code{.plotInitialTime} or \code{.saveInitialTime}, then
these arguments will not do anything.

If \code{cache} is TRUE, this allows for a seamless way to "save" results
of a simulation. The  user does not have to intentionally do any saving manually.
Instead, upon a call to \code{spades} in which the simList is identical,
the function will simply return the result that would have come if it had
been rerun. Use this with caution, as it will return exactly the result from
a previous run, even if there is stochasticity internally.
Caching is only based on the input simList. See also \code{experiment} for
the same mechanism, but it can be used with replication.
See also the vignette on caching for examples.

If \code{debug} is specified, it can be a logical or character vector.
If not specified, the package option \code{spades.debug} is used.
In all cases, something will be printed to the console immediately before each
event is being executed.
If \code{TRUE}, then the event immediately following will be printed as it
runs (equivalent to \code{current(sim)}).
If a character string, then it can be one of the many \code{simList} accessors,
such as \code{events}, \code{params}, \code{"simList"} (print the entire simList),
or any R expression.
If an R expression it will be evaluated with access to the \code{sim} object.
If this is more than one character string, then all will be printed to the
screen in their sequence.
}
\note{
The debug option is primarily intended to facilitate building simulation
models by the user.
Will print additional outputs informing the user of updates to the values of
various \code{simList} slot components.
See \url{https://github.com/PredictiveEcology/SpaDES/wiki/Debugging} for details.
}
\examples{
\dontrun{
mySim <- simInit(
 times = list(start = 0.0, end = 2.0, timeunit = "year"),
 params = list(
   .globals = list(stackName = "landscape", burnStats = "nPixelsBurned")
 ),
 modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
 paths = list(modulePath = system.file("sampleModules", package = "SpaDES.core"))
)
spades(mySim)

# set default debug printing for the current session
# setOption(spades.debug = TRUE)

# Different debug options (overrides the package option 'spades.debug')
spades(mySim, debug = TRUE) # Fastest
spades(mySim, debug = "simList")
spades(mySim, debug = "print(table(sim$landscape$Fires[]))")

# Can turn off plotting, and inspect the output simList instead
out <- spades(mySim, .plotInitialTime = NA) # much faster
completed(out) # shows completed events

# use cache -- simInit should generally be rerun each time a spades call is made
#   to guarantee that it is identical. Here, run spades call twice, first
#   time to establish cache, second time to return cached result
for (i in 1:2) {
 mySim <- simInit(
   times = list(start = 0.0, end = 2.0, timeunit = "year"),
   params = list(
     .globals = list(stackName = "landscape", burnStats = "nPixelsBurned")
   ),
   modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
   paths = list(modulePath = system.file("sampleModules", package = "SpaDES.core"))
 )
 print(system.time(out <- spades(mySim, cache = TRUE)))
}
}

}
\references{
Matloff, N. (2011). The Art of R Programming (ch. 7.8.3).
            San Fransisco, CA: No Starch Press, Inc..
            Retrieved from \url{https://www.nostarch.com/artofr.htm}
}
\seealso{
\code{\link{SpaDES.core-package}},
\code{\link{experiment}} for using replication with \code{spades},
\code{\link{simInit}}, and the caching vignette (very important for reproducibility):
\url{https://CRAN.R-project.org/package=SpaDES/vignettes/iii-cache.html} which
uses \code{\link[reproducible]{Cache}}.
}
\author{
Alex Chubaty and Eliot McIntire
}
