% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interGradientsDL.R
\name{interGradientsDL}
\alias{interGradientsDL}
\title{Calculate gradients of predicted cell types/loss function with respect to
input features for interpreting trained deconvolution models}
\usage{
interGradientsDL(
  object,
  method = "class",
  normalize = TRUE,
  scaling = "standardize",
  verbose = TRUE
)
}
\arguments{
\item{object}{\code{\linkS4class{SpatialDDLS}} object containing a trained
deconvolution model (\code{trained.model} slot) and pure mixed
transcriptional profiles (\code{mixed.profiles} slot).}

\item{method}{Method to calculate gradients with respect to inputs. It can be
\code{'class'} (gradients of predicted classes w.r.t. inputs),
\code{'loss'} (gradients of loss w.r.t. inputs) or \code{'both'}.}

\item{normalize}{Whether to normalize data using logCPM (\code{TRUE} by
default). This parameter is only considered when the method used to
simulate the mixed transcriptional profiles (\code{simMixedProfiles}
function) was \code{"AddRawCount"}. Otherwise, data were already
normalized. This parameter should be set according to the transformation
used to train the model.}

\item{scaling}{How to scale data. It can be: \code{"standardize"}
(values are centered around the mean with a unit standard deviation),
\code{"rescale"} (values are shifted and rescaled so that they end up
ranging between 0 and 1, by default) or \code{"none"} (no scaling is
performed). This parameter should be set according to the transformation
used to train the model.}

\item{verbose}{Show informative messages during the execution (\code{TRUE} by
default).}
}
\value{
Object containing gradients in the \code{interpret.gradients} slot of
the \code{DeconvDLModel} object (\code{trained.model} slot).
}
\description{
This function enables users to gain insights into the interpretability of the
deconvolution model. It calculates the gradients of classes/loss function
with respect to the input features used in training. These numeric values are
calculated per gene and cell type in pure mixed transcriptional profiles,
providing information on the extent to which each feature influences the
model's prediction of cell proportions for each cell type.
}
\details{
Gradients of classes / loss function with respect to the input features are
calculated exclusively using pure mixed transcriptional profiles composed of
a single cell type. Consequently, these numbers can be interpreted as the
extent to which each feature is being used to predict each cell type
proportion. Gradients are calculated at the sample level for each gene, but
only mean gradients by cell type are reported. For additional details, see
Mañanes et al., 2023.
}
\examples{
\donttest{
set.seed(123)
sce <- SingleCellExperiment::SingleCellExperiment(
  assays = list(
    counts = matrix(
      rpois(30, lambda = 5), nrow = 15, ncol = 10,
      dimnames = list(paste0("Gene", seq(15)), paste0("RHC", seq(10)))
    )
  ),
  colData = data.frame(
    Cell_ID = paste0("RHC", seq(10)),
    Cell_Type = sample(x = paste0("CellType", seq(2)), size = 10,
                       replace = TRUE)
  ),
  rowData = data.frame(
    Gene_ID = paste0("Gene", seq(15))
  )
)
SDDLS <- createSpatialDDLSobject(
  sc.data = sce,
  sc.cell.ID.column = "Cell_ID",
  sc.gene.ID.column = "Gene_ID",
  sc.filt.genes.cluster = FALSE
)
SDDLS <- genMixedCellProp(
  object = SDDLS,
  cell.ID.column = "Cell_ID",
  cell.type.column = "Cell_Type",
  num.sim.spots = 50,
  train.freq.cells = 2/3,
  train.freq.spots = 2/3,
  verbose = TRUE
)
SDDLS <- simMixedProfiles(SDDLS)
SDDLS <- trainDeconvModel(
  object = SDDLS,
  batch.size = 12,
  num.epochs = 5
)
## calculating gradients
SDDLS <- interGradientsDL(SDDLS)
}
  
}
\seealso{
\code{\link{deconvSpatialDDLS}} \code{\link{plotTrainingHistory}}
}
