#
# e2e_plot_mc_parms.R
#
#' Plot diagnostic data from sensitivity and Monte Carlo analyses.
#'
#' The function generates diagnostic plots from either a parameter sensitivity analysis or a Monte Carlo simulation of credible intervals of model outputs.
#'
#' In both cases, the inputs required are a model list object created by the e2e_read() function defining the model configuration and relevant model.ident
#' argument to point to the required files, and a logical argument to determne the origin of data files (either saved as csv files in the user workspace, or
#' example data for the North Sea model provided with the package.
#'
#' \strong{Details relating to sensitivity analysis plot:}
#'
#' The function reads processed results generated by the function e2e_run_sens(), or the function e2e_process_sens_mc(...,selection="SENS"), and plots the Elementary Effect mean (x-axis; magnitude of sensitivity)
#' against Elementary Effect standard deviation (y-axis; strength of interactions between parameters).
#'
#' Processed results from the function e2e_run_sens(), or the function e2e_process_sens_mc(...,selection="SENS"), are stored in the csv file
#' ../Modelname/Variantname/sorted_parameter_elementary_effects-*.csv 
#' where * represents the model run identifier (model.ident) text embedded in the R-list object created by the e2e_read() function. The path to this file
#' is relative to the results folder specified in teh e2e_read() function call.
#'
#' Each symbol in the plot represents a single parameter in the model. The parameters are colour-coded to indicate 6 different types - fitted and fixed
#' parameters of the ecology model, fishing fleet model parameters, fishery harvest ratios, environmental drivers, and physical configuration parameters.
#'
#' The plot also shows a wedge formed by the two dashed lines. These correspond to +/-2 standard errors of the mean, so for points falling outside of the
#' wedge there is a significant expectation that the distribution of elementary effects is non-zero. For points falling within the wedge the distribution of
#' elementary effects is not significantly different from zero.
#' 
#' For details of how the Elementary Effect values are derived for each parameter see help(e2e_run_sens)
#'
#' \strong{Details relating to Monte Carlo analysis plot:}
#'
#' The function creates a plot showing the credible distributions of ecology model parameters based on the results from the e2e_run_mc() function.
#' These distributions are formed from the input distributions to the Monte Carlo process, weighted by the likelihood of observed target data
#' on the state of the ecosystem given each combination of parameter values.
#'
#' Post-processed data from the e2e_run_mc() function are stored in the file
#' ../Modelname/Variantname/CredInt/CredInt_processed_PARAMETERresults_data-*.csv, where * represents the model run identifier
#' (model.ident) text embedded in the R-list object created by the e2e_read() function. The path to this file
#' is relative to the results folder specified in teh e2e_read() function call.
#'
#' Each parameter in the plot is scaled to its baseline value as (value - baseline)/baseline. Ideally, the baseline is the maximum likelihood model
#' developed by application of the optimization functions available in the package (e.g. e2e_optimize_eco()).
#' Each parameter is then represented by a box and whisker plot which shows the distribution of credible parameter values around zero, i.e.
#' around the baseline. The median of the credible values distribution for each parameter
#' is shown my a black tick-mark. The box spans the 50% credible interval (quartiles of the cumulative likelihood
#' of simulated values). Whisker lines span the 99% credible interval.
#'
#' The individual parameters are identified by numbers (rather than text names). These numbers correspond to the column numnbers in
#' the file ../Modelname/Variantname/CredInt/CredInt_processed_parameters-*.csv . Details of the parameters associated with each identification
#' number are available as a dataframe by using the function e2e_get_parmdoc().
#' 
#' The input distribution of parameter values to the Monte Carlo process is drawn from a random uniform distribution with limits specified in
#' the monte_carlo control file for the model setup (located in a sub-folder of the /Param/control folder). This distribution is shown by
#' a red box and whisker at the bottom of the plot. Given the random uniform input we expect the quartiles (limits of the box) to be symetrical
#' and located mid-way between zero and the upper and lower extremes. Vertical red lines show the expected limits of the quartiles boxes if 
#' model results were completely insensitive to individual parameter values.
#'
#' The extent to which individual parameter distributions deviate from the random uniform input is an indication of their sensitivity in the model. 
#' Parameters whose distributions are tightly confined around zero (the baseline value) are highly sensitive.
#'
#' For some parameters, in particular the preference parameters, their credible distributions may span a wider range than the inputs. This may seem
#' unexpected, but arises because within each feeding guild the preference parameters are not independent of each other. The preferences within each guild
#' must sum to 1. Hence, during the Monte Carlo process, after drawing new values of the preference values they are all rescaled to sum to 1, which may mean that
#' some of them will have been varied by more than the original coefficient of variation of the input random uniform. 
#'
#' For details of how the distribution of credible output values from StrathE2E are calculated see the help information for the e2e_run_mc() function.
#'
#' @references For details on the sensitivity analysis methodology see: Morris, M.D. (1991). Factorial sampling plans for preliminary computational experiments. Technometrics, 33, 161-174.
#'
#' @param model R-list object defining the model configuration, compiled by the e2e_read() function
#' @param selection Text string from a list identifying source of data to bed merged. Select from: "SENS", "MC", referring to sensitivity analysis or Monte Carlo analysis. Remember to include the phrase within "" quotes.
#' @param use.example Logical. If TRUE use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE).
#'
#' @return Graphical display in a new graphics window.
#'
#' @seealso \code{\link{e2e_read}}, \code{\link{e2e_run_mc}}, \code{\link{e2e_run_sens}}, \code{\link{e2e_get_parmdoc}} 
#'
#' @export
#'
#' @examples
#' # These examples requires the Strathe2E2examples supplementary data package.
#'
#' # Load the 1970-1999 version of the North Sea model supplied with the package:
#' if(require(StrathE2E2examples)){
#'     model <- e2e_read("North_Sea", "1970-1999")
#' # Plotting illustrated here using the example data sets, but you could run the 
#' # functions e2e_run_sens() or e2e_run_mc() to generate some results instead.
#' 
#' # Plot the results of a parameter sensitivity analysis. Each point represents one parameter.
#' # The x-axis represents the sensitivity of the overall likelihood of the observed data to 
#' # The y-axis represents the extent to which each parameter interacts with others.
#'     e2e_plot_sens_mc(model, selection="SENS", use.example=TRUE)  
#' }
#'
#' #---------------------------------------------------------------------------
#'
#' # Plot the credible distributions of ecology model parameters based on the results of a
#' # Monte Carlo analysis. These distributions are formed from the input distributions to 
#' # the Monte Carlo process, weighted by the likelihood of observed target data
#' # on the state of the ecosystem given each combination of parameter values.
#' if(require(StrathE2E2examples)){
#'     e2e_plot_sens_mc(model, selection="MC", use.example=TRUE)    # for Monte Carlo results
#'
#' # To direct the graph output to a file rather than the screen, wrap the
#' # plot_Monte_Carlo_parameter_distributions() function call in a graphical device call:
#' # or jpeg("plot.jpg"), png("plot.png")
#'     pdf(file.path(tempdir(), "plot.pdf"),width=9,height=4)
#'     e2e_plot_sens_mc(model, selection="SENS", use.example=TRUE)
#'     dev.off()
#' }
#'
#' #---------------------------------------------------------------------------
#'
#' # For a comprehensive workflow through the generation of sensitivity and
#' # Monte Carlo analysis data see:
#'     help(e2e_run_sens)
#'     help(e2e_run_mc)
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------

e2e_plot_sens_mc <- function(model, selection="SENS", use.example=FALSE) {

   oo <- options()
   on.exit(options(oo))

	if (use.example == TRUE) hasExamples()

start_par = par()$mfrow
on.exit(par(mfrow = start_par))


if(selection=="SENS"){

	plot_sensitivity_results(model=model, use.example=use.example)

} else if(selection=="MC"){

	plot_mc_parameter_likelihoods(model=model, use.example=use.example)

} else {
	stop("Error: unknown plot selection '", selection, "' !\n")
}



}

