% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rips_complexes.R
\name{vr_graphs}
\alias{vr_graphs}
\title{Compute Vietoris-Rips graphs of a dataset at particular epsilon radius values.}
\usage{
vr_graphs(X, distance_mat = FALSE, eps, return_clusters = TRUE)
}
\arguments{
\item{X}{either a point cloud data frame/matrix, or a distance matrix.}

\item{distance_mat}{a boolean representing if the input `X` is a distance matrix, default value is `FALSE`.}

\item{eps}{a numeric vector of the positive scales at which to compute the Rips-Vietoris complexes, i.e. all edges at most the specified values.}

\item{return_clusters}{a boolean determining if the connected components (i.e. data clusters) of the complex should be explicitly returned, default is `TRUE`.}
}
\value{
A list with a `vertices` field, containing the rownames of `X`, and then a list `graphs` one (named) entry for each value in `eps`. Each entry is a list with a `graph` field, storing the (undirected) edges in the Rips-Vietoris complex in matrix format, and a `clusters` field, containing vectors of the data indices (or row names) in each connected component of the Rips graph.
}
\description{
Persistence diagrams computed from Rips-Vietoris filtrations contain information about 
distance radius scales at which topological features of a dataset exist, but the features
can be challenging to visualize, analyze and interpret. In order to help solve this problem the `vr_graphs`
function computes the 1-skeleton (i.e. graph) of Rips complexes at particular radii, called "Vietoris-Rips graphs" (VR graphs) in the literature.
}
\details{
This function may be used in conjunction with the igraph package to visualize the graphs (see \code{\link{plot_vr_graph}}).
}
\examples{

if(require("TDA") & require("igraph"))
{
  # simulate data from the unit circle and calculate 
  # its diagram
  df <- TDA::circleUnif(n = 25)
  diag <- TDA::ripsDiag(df,maxdimension = 1,maxscale = 2)
  
  # get minimum death radius of any data cluster
  min_death_H0 <- 
  min(diag$diagram[which(diag$diagram[,1] == 0),3L])
  
  # get birth and death radius of the loop
  loop_birth <- as.numeric(diag$diagram[nrow(diag$diagram),2L])
  loop_death <- as.numeric(diag$diagram[nrow(diag$diagram),3L])

  # compute VR graphs at radii half of 
  # min_death_H0 and the mean of loop_birth and 
  # loop_death, returning clusters
  graphs <- vr_graphs(X = df,eps = 
  c(0.5*min_death_H0,(loop_birth + loop_death)/2))

  # verify that there are 25 clusters for the smaller radius
  length(graphs$graphs[[1]]$clusters)
  
}
}
\references{
A Zomorodian, The tidy set: A minimal simplicial set for computing homology of clique complexes in Proceedings of the Twenty-Sixth Annual Symposium on Computational Geometry, SoCG ’10. (Association for Computing Machinery, New York, NY, USA), p. 257–266 (2010).
}
\author{
Shael Brown - \email{shaelebrown@gmail.com}
}
