% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeAlgebraicFunctions}
\alias{computeAlgebraicFunctions}
\title{Compute Algebraic Functions from a Persistence Diagram}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeAlgebraicFunctions()} computes the following four algebraic functions based on the birth and death values:

\itemize{
  \item \eqn{f_1=\sum_i b_i (d_i - b_i).}
  \item  \eqn{f_2=\sum_i (d_{\max} - d_i) (d_i - b_i)}, where \eqn{d_{\max} = \max(d_i)}.
  \item \eqn{f_3=\sum_i b_i^2 (d_i - b_i)^4}.
  \item \eqn{f_4=\sum_i (d_{\max} - d_i)^2 (d_i - b_i)^4}.
}
Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computeAlgebraicFunctions(D, homDim)
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}
\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}
}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes the four algebraic functions based on the filtered data. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.
}


\value{
A (named) numeric vector \eqn{(f_1,f_2,f_3,f_4)}.
}

\author{Umar Islambekov}

\references{
1. Adcock, A., Carlsson, E. and Carlsson, G., 2013. The ring of algebraic functions on persistence bar codes. Homology, Homotopy Appl., 18:381–402, 2016.

2. Ali, D., Asaad, A., Jimenez, M.J., Nanda, V., Paluzo-Hidalgo, E. and Soriano-Trigueros, M., (2023). A survey of vectorization methods in topological data analysis. IEEE Transactions on Pattern Analysis and Machine Intelligence.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

# Compute algebraic functions for homological dimension H_0
computeAlgebraicFunctions(D, homDim = 0)

# Compute algebraic functions for homological dimension H_1
computeAlgebraicFunctions(D, homDim = 1)
}
