% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeBettiCurve}
\alias{computeBettiCurve}
\title{A Vector Summary of the Betti Curve}

\description{
For a given persistence diagram \eqn{D={(b_i,d_i)}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeBettiCurve()} vectorizes the Betti curve \deqn{\beta(t)=\sum_{i=1}^N\bold 1_{[b_i,d_i)}(t)} based on a scale sequence \code{scaleSeq}. The evaluation method depends on the argument \code{evaluate}.
}

\usage{
computeBettiCurve(D, homDim, scaleSeq, evaluate = "intervals")
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}

\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}

\item{scaleSeq}{a numeric vector of increasing scale values used for vectorization.}

\item{evaluate}{a character string indicating the evaluation method. Must be either \code{"intervals"} (default) or \code{"points"}.}

}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes values based on the filtered data and \code{scaleSeq}. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.
}

\value{
A numeric vector containing elements computed using \code{scaleSeq}=\eqn{\{t_1,t_2,\ldots,t_n\}} according to the method specified by \code{evaluate}.

\itemize{
  \item \code{"intervals"}: Computes average values of the Betti curve over intervals defined by consecutive elements in \code{scaleSeq}:

  \deqn{\Big(\frac{1}{\Delta t_1}\int_{t_1}^{t_2}\beta(t)dt,\frac{1}{\Delta t_2}\int_{t_2}^{t_3}\beta(t)dt,\ldots,\frac{1}{\Delta t_{n-1}}\int_{t_{n-1}}^{t_n}\beta(t)dt\Big)\in\mathbb{R}^{n-1},}
where \eqn{\Delta t_k=t_{k+1}-t_k}.

  \item \code{"points"}: Computes values of the Betti curve at each point in \code{scaleSeq}:

  \deqn{(\beta(t_1),\beta(t_2),\ldots,\beta(t_n))\in\mathbb{R}^n.}
}
}

\author{Umar Islambekov, Hasani Pathirana}

\references{
1. Chazal, F., & Michel, B. (2021). An Introduction to Topological Data Analysis: Fundamental and Practical Aspects for Data Scientists. Frontiers in Artificial Intelligence, 108.

2. Chung, Y. M., & Lawson, A. (2022). Persistence curves: A canonical framework for summarizing persistence diagrams. Advances in Computational Mathematics, 48(1), 1-42.

3. Islambekov, U., & Pathirana. H. (2024). Vector Summaries of Persistence Diagrams for Permutation-based Hypothesis Testing. Foundations of Data Science 6 (1), 41-61.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

scaleSeq = seq(0, 2, length.out = 11) # A sequence of scale values

# Compute a vector summary of the Betti curve for homological dimension H_0
computeBettiCurve(D, homDim = 0, scaleSeq)

# Compute a vector summary of the Betti curve for homological dimension H_1
computeBettiCurve(D, homDim = 1, scaleSeq)
}
