#' @title Summary Statistics Extraction
#' @description Extract summary data needed for other functions from raw data.
#' @param x Tibble/data frame containing raw data.
#' @param Sex Number of the column containing sex 'M' for male and 'F' for female, Default: 1
#' @param Pop Number of the column containing populations' names, Default: 2
#' @param firstX Number of column containing measured parameters (First of
#'   multiple in case of multivariate analysis), Default: 3
#' @param test `1` for Greene t-test [Tg], `2` for [univariate], `3` for sex
#'   specific ANOVA [aovSS], and `4` for [multivariate],
#'   Default: 1
#' @param run Logical; if TRUE runs the corresponding test after data
#'   extraction, Default: TRUE
#' @param ... Additional arguments that could be passed to the test of choice
#' @return Input for other functions.
#' @details Raw data is entered in a wide format tibble/data frame similar to
#'   [Howells] data set. The first two columns contain sex `Sex` (`M` for
#'   male and `F` for female) (Default: `1`) and populations' names `Pop`
#'   (Default: `2`). Starting from `firstX` column (Default: `3`), measured
#'   parameters are entered each in a separate column.
#' @examples
#' # for multivariate test
#' library(TestDimorph)
#' extract_sum(Howells,test=4)
#' # for univariate test on a specific parameter
#' library(TestDimorph)
#' extract_sum(Howells, test = 2,firstX = 4)
#'
#' @rdname extract_sum
#' @export
#' @importFrom dplyr group_by filter summarise_all select ungroup %>% full_join
#' @importFrom Rfast pooled.cov
#' @importFrom stats sd
#' @importFrom tibble is_tibble
#' @importFrom rlang abort
extract_sum <-
    function(x,
             Sex = 1,
             Pop = 2,
             firstX = 3,
             test = 1,
             run = TRUE,
             ...) {
        if (!(is.data.frame(x) || tibble::is_tibble(x))) {
            rlang::abort("x must be a tibble or a dataframe")
        }
        x <- data.frame(x)
        if (!(Sex %in% seq_along(1:ncol(x))))   {
            rlang::abort("Sex should be number from 1 to ncol(x)")

        }
        if (!(Pop %in% seq_along(1:ncol(x))))   {
            rlang::abort("Pop should be number from 1 to ncol(x)")

        }
        if (!(firstX %in% seq_along(1:ncol(x))))   {
            rlang::abort("firstX should be number from 1 to ncol(x)")

        }
        if (!(run %in% c(TRUE, FALSE)))   {
            rlang::abort("run should be either TRUE or FALSE")

        }
        x$Pop <- x[, Pop]
        x$Sex <- x[, Sex]
        x$Pop <- factor(x$Pop)
        x$Sex <- factor(x$Sex)
        if (length(unique(x$Sex)) != 2) {
            rlang::abort("Sex column should be a factor with only 2 levels `M` and `F`")
        }
        if (!(test %in% 1:4)) {
            rlang::abort("Test can only be be from 1 to 4")

        }
        if (test == 4) {
            x <- as.data.frame.list(x)
            sex <- as.numeric(x[, Sex]) - 1
            pop <- as.numeric(x[, Pop])
            pop.names <- names(table(x[, Pop]))
            N.pops <- length(pop.names)
            ina <- pop + N.pops * sex
            X <- x[, -(1:(firstX - 1))]
            Trait.names <- colnames(X)
            V <- Rfast::pooled.cov(as.matrix(X), ina)
            D <- diag(1 / sqrt(diag(V)))
            R.res <- D %*% V %*% D
            M.mu <-
                x %>%  group_by(Pop) %>% filter(Sex == "M") %>% select(firstX:ncol(x))  %>% summarise_all(.funs =
                                                                                                              mean) %>% ungroup() %>% data.frame() %>% select(-1) %>% as.matrix()
            row.names(M.mu) <- pop.names
            F.mu <-
                x %>%  group_by(Pop) %>% filter(Sex == "F") %>% select(firstX:ncol(x))  %>% summarise_all(.funs =
                                                                                                              mean)  %>% ungroup() %>%  data.frame() %>%  select(-1) %>% as.matrix()
            row.names(F.mu) <- pop.names

            m <- table(x[, Pop][x[, Sex] == "M"])
            f <- table(x[, Pop][x[, Sex] == "F"])

            F.sdev <-
                matrix(NA, nrow = N.pops, ncol = NCOL(x) - firstX + 1)
            for (i in 1:N.pops) {
                F.sdev[i, ] <- apply(X[ina == i, ], 2, stats::sd)
            }

            row.names(F.sdev) <- pop.names
            colnames(F.sdev) <- Trait.names

            M.sdev <-
                matrix(NA, nrow = N.pops, ncol = NCOL(x) - firstX + 1)
            for (i in 1:N.pops) {
                M.sdev[i, ] <- apply(X[ina == N.pops + i, ], 2, stats::sd)
            }

            row.names(M.sdev) <- pop.names
            colnames(M.sdev) <- Trait.names

            v <-
                list(
                    R.res = R.res,
                    M.mu = M.mu,
                    F.mu = F.mu,
                    m = m,
                    f = f,
                    M.sdev = M.sdev,
                    F.sdev = F.sdev
                )
            if (run == TRUE) {
                return(multivariate(v, ...))
            } else {
                return(v)
            }
        } else {
            M <-
                x %>% filter(Sex == "M") %>% group_by(Pop) %>% select(firstX) %>%  summarise_all(list(
                    M.mu = mean,
                    M.sdev = sd,
                    m = length
                ))
            F <-
                x %>% filter(Sex == "F") %>% group_by(Pop) %>% select(firstX) %>%  summarise_all(list(
                    F.mu = mean,
                    F.sdev = sd,
                    f = length
                ))
            df <- dplyr::full_join(M, F, by = "Pop")
            if (test == 2) {
                if (run == TRUE) {
                    return(univariate(x = df, Pop = 1, ...))
                } else {
                    return(df)
                }
            }
            if (test == 3) {
                if (run == TRUE) {
                    return(aovSS(x = df, Pop = 1, ...))
                } else {
                    return(df)
                }
            }
            if (test == 1) {
                if (run == TRUE) {
                    return(Tg(x = df, Pop = 1, ...))
                } else {
                    return(df)
                }
            }
        }
    }
