#' Using double list to visit vector binary tree
#'
#' @description Advanced visiting for the vector binary tree. Return a double list by specific assigment determined by
#' the argument \code{inq}.
#' @param x The vector binary tree to be visited. Traversal is acheivable through invalid assignment in desired layer.
#' @param inq An integer double list to determine the location to be visited. The length of \code{inq} should be the same
#' as the layers of visited vector binary tree, while all elements in vector in each layer of \code{inq} should not over
#' the intrinsic length of visited vector binary tree layer, otherwise all elements will be returned in this layer.
#'
#' @return Return a double list according to the argument \code{inq}.
#' @export advbtinq
#' @seealso \code{\link[VBTree:vbtinq]{vbtinq}}, \code{\link[VBTree:vbtsub]{vbtsub}}, \code{\link[VBTree:advbtsub]{advbtsub}}.
#'
#' @examples
#' #Make vector binary tree:
#' colnamevbt <- dl2vbt(chrvec2dl(colnames(datatest)))
#'
#' #Visit by specific assignment:
#' visit <- list(c(2), c(3:6), c(2,4), 1)
#' advbtinq(colnamevbt, visit)
#'
#' #Traversal of the second layers:
#' visit <- list(c(2), colnamevbt$dims[2]+1, c(2,4), 1)
#' advbtinq(colnamevbt, visit)
#'
#' #Invalid assignments in 1st and 3rd layers:
#' visit <- list(c(3), c(3:6), c(5), 1)
#' advbtinq(colnamevbt, visit)
#' @keywords Vector.Binary.Tree Double.List
advbtinq <- function(x, inq){
  # input diagnose
  if(class(x)!="Vector.Binary.Tree"){
    stop("x should be a Vector.Binary.Tree generated by arr2vbt(), ts2vbt() or dl2vbt().", call. = FALSE)
  }
  dims <- x$dims
  layers <- length(dims)

  # this inquiry method using double list: inq is a double list and desired elements to be inquired should be save in each layers in form of vector (integer, numeric)
  if(is.vector(inq)!=TRUE | length(inq)!=layers){
    stop("inq must be a vector with the same size as the layers of x.", call. = FALSE)
  }

  # diagnose for available dimension assignment;
  # overflowed layers will be replaced by full sequence.
  j <- 1
  prt <- rep(TRUE, layers)
  while (j <= layers) {
    check.inq <- inq[[j]]
    if(any(check.inq > dims[j])){
      i <- which.max(check.inq > dims[j])
      warning(paste("The ", i, " assignment(s) in layer ", j," is overflow. All elements in layer ", j, " will be returned completely.", sep = ""), call. = FALSE)
      inq[[j]] <- -1
      prt[j] <- FALSE
    }
    j <- j + 1
  }

  x_dl <- vbt2dl(x)
  result <- x_dl

  i <- 1
  while (i <= layers) {
    if(prt[i]){
      result[[i]] <- x_dl[[i]][inq[[i]]]
    } else {
      result[[i]] <- x_dl[[i]]
    }
    i <- i + 1
  }
  return(result)
}


#' Using double list to generate sub tree from vector binary tree
#'
#' @description Advanced visiting for the vector binary tree. Generating a sub tree from visited vector binary tree,
#' through specific assigment determined by the argument \code{inq}.
#' @param x The vector binary tree to be visited. Traversal is acheivable through invalid assignment in desired
#' layers.
#' @param inq An integer double list to determine the visiting location. The length of \code{inq} should be the same
#' as the layers of visited vector binary tree. If any assign element in specificed layer exceeds its intrinsic
#' length of visited vector binary tree layer, all elements will be returned in this layer.
#'
#' @return Return a sub tree from visited vector binary tree, according to the argument \code{inq}.
#' @export advbtsub
#' @seealso \code{\link[VBTree:vbtinq]{vbtinq}}, \code{\link[VBTree:vbtsub]{vbtsub}}, \code{\link[VBTree:advbtinq]{advbtinq}}.
#'
#' @examples
#' #Make vector binary tree:
#' colnamevbt <- dl2vbt(chrvec2dl(colnames(datatest)))
#'
#' #Visit by specific assignment:
#' visit <- list(c(2), c(3:6), c(2,4), 1)
#' advbtsub(colnamevbt, visit)
#'
#' #Traversal of the second layers:
#' visit <- list(c(2), colnamevbt$dims[2]+1, c(2,4), 1)
#' advbtsub(colnamevbt, visit)
#'
#' #Invalid assignments in 1st and 3rd layers:
#' visit <- list(c(3), c(3:6), c(5), 1)
#' advbtsub(colnamevbt, visit)
#' @keywords Vector.Binary.Tree
advbtsub <- function(x, inq){
  result <- advbtinq(x, inq)
  result <- dl2vbt(result)
  return(result)
}
