/******************************************************************************
 * @section DESCRIPTION
 *
 * Initialize parameters structure.
 *
 * @section LICENSE
 *
 * The Variable Infiltration Capacity (VIC) macroscale hydrological model
 * Copyright (C) 2016 The Computational Hydrology Group, Department of Civil
 * and Environmental Engineering, University of Washington.
 *
 * The VIC model is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *****************************************************************************/

#include <vic_driver_shared_all.h>

/******************************************************************************
 * @brief    Initialize parameters structure.
 *****************************************************************************/
void
initialize_parameters(void)
{
    extern parameters_struct param;
    // Initialize temporary parameters

    // Lapse Rate
    param.LAPSE_RATE = -0.0065;

    // Precipitation Guage Height
    param.GAUGE_HEIGHT = 1.0;

    // Huge Resistance Term
    param.HUGE_RESIST = 1e20;

    // Surface Albedo Parameters
    param.ALBEDO_BARE_SOIL = 0.2;

    // Surface Emissivities
    param.EMISS_GRND = 0.97;
    param.EMISS_VEG = 0.97;
    param.EMISS_ICE = 0.97;
    param.EMISS_SNOW = 0.97;
    param.EMISS_H2O = 0.98;

    // Soil Constraints
    param.SOIL_RARC = 100.0;
    param.SOIL_RESID_MOIST = 0.0;
    param.SOIL_SLAB_MOIST_FRACT = 1.0;
    param.SOIL_WINDH = 10.0;

    // Vegetation Parameters
    param.VEG_LAI_SNOW_MULTIPLIER = 0.0005;
    param.VEG_LAI_WATER_FACTOR = 0.1;
    param.VEG_MIN_INTERCEPTION_STORAGE = 0.005;
    param.VEG_RATIO_DH_HEIGHT = 0.67;
    param.VEG_RATIO_RL_HEIGHT = 0.123;

    // Canopy Parameters
    param.CANOPY_CLOSURE = 4000.0;
    param.CANOPY_RSMAX = 5000.0;
    param.CANOPY_VPDMINFACTOR = 0.1;

    // Lake Parameters
    param.LAKE_TMELT = 0.0;
    param.LAKE_MAX_SURFACE = 0.6;
    param.LAKE_BETA = 0.001;
    param.LAKE_FRACMIN = 0.10;
    param.LAKE_FRACLIM = 0.02;
    param.LAKE_DM = 1.38889E-07;
    param.LAKE_SNOWCRIT = 0.05;
    param.LAKE_ZWATER = 0.0045;
    param.LAKE_ZSNOW = 0.005;
    param.LAKE_RHOSNOW = 250;
    param.LAKE_CONDI = 2.3;
    param.LAKE_CONDS = 0.7;
    param.LAKE_LAMISW = 1.5;
    param.LAKE_LAMILW = 20.0;
    param.LAKE_LAMSSW = 6.0;
    param.LAKE_LAMSLW = 20.0;
    param.LAKE_LAMWSW = 0.3;
    param.LAKE_LAMWLW = 1.4;
    param.LAKE_A1 = 0.7;
    param.LAKE_A2 = 0.3;
    param.LAKE_QWTAU = 43200.0;
    param.LAKE_MAX_ITER = 50;

    // Saturation Vapor Pressure Parameters
    param.SVP_A = 0.61078;
    param.SVP_B = 17.269;
    param.SVP_C = 237.3;

    // Photosynthesis Parameters
    param.PHOTO_OMEGA = 0.12;
    param.PHOTO_LAIMAX = 8.0;
    param.PHOTO_LAILIMIT = 3.0;
    param.PHOTO_LAIMIN = 1.0e-9;
    param.PHOTO_EPAR = 2.2e5;
    param.PHOTO_FCMAX = 0.9;
    param.PHOTO_FCMIN = 1.0e-3;
    param.PHOTO_ZENITHMIN = 0.0174524;
    param.PHOTO_ZENITHMINPAR = 1.0e-3;
    param.PHOTO_ALBSOIPARMIN = 0.0;
    param.PHOTO_MINMAXETRANS = 1.0e-12;
    param.PHOTO_MINSTOMCOND = 0.0;
    param.PHOTO_FCI1C3 = 0.87;
    param.PHOTO_FCI1C4 = 0.67;
    param.PHOTO_OX = 0.21;
    param.PHOTO_KC = 460.0e-6;
    param.PHOTO_KO = 330.0e-3;
    param.PHOTO_EC = 59356.0;
    param.PHOTO_EO = 35948.0;
    param.PHOTO_EV = 58520.0;
    param.PHOTO_ER = 45000.0;
    param.PHOTO_ALC3 = 0.28;
    param.PHOTO_FRDC3 = 0.011;
    param.PHOTO_EK = 50967.0;
    param.PHOTO_ALC4 = 0.04;
    param.PHOTO_FRDC4 = 0.042;
    param.PHOTO_THETA = 0.83;
    param.PHOTO_FRLEAF = 0.4;
    param.PHOTO_FRGROWTH = 0.25;

    // Soil Respiration Parameters
    param.SRESP_E0_LT = 308.56;
    param.SRESP_T0_LT = 227.13;
    param.SRESP_WMINFM = 0.0;
    param.SRESP_WMAXFM = 1.0;
    param.SRESP_WOPTFM = 0.5;
    param.SRESP_RHSAT = 0.15;
    param.SRESP_RFACTOR = 0.5;
    param.SRESP_TAULITTER = 2.86;
    param.SRESP_TAUINTER = 33.3;
    param.SRESP_TAUSLOW = 1000.0;
    param.SRESP_FAIR = 0.7;
    param.SRESP_FINTER = 0.985;

    // Snow Parameters
    param.SNOW_MAX_SURFACE_SWE = 0.125;
    param.SNOW_LIQUID_WATER_CAPACITY = 0.035;
    param.SNOW_NEW_SNOW_DENSITY = 50.0;
    param.SNOW_NEW_SNOW_DENS_MAX = 400.0;
    param.SNOW_DEPTH_THRES = 1.e-8;
    param.SNOW_DENS_DMLIMIT = 100.0;
    param.SNOW_DENS_DMLIMIT_FACTOR = 1.15;
    param.SNOW_DENS_MAX_CHANGE = 0.9;
    param.SNOW_DENS_ETA0 = 3.6e6;
    param.SNOW_DENS_C1 = 0.04;
    param.SNOW_DENS_C2 = 2.778e-6;
    param.SNOW_DENS_C3 = 1.0;
    param.SNOW_DENS_C3_CONST = -0.046;
    param.SNOW_DENS_C4 = 1.0;
    param.SNOW_DENS_C4WET = 2.0;
    param.SNOW_DENS_C5 = 0.08;
    param.SNOW_DENS_C6 = 0.021;
    param.SNOW_DENS_F = 0.6;
    param.SNOW_DENS_EXP = 0.35;
    param.SNOW_DENS_DENOM = 10.;
    param.SNOW_NEW_SNT_C1 = 67.92;
    param.SNOW_NEW_SNT_C2 = 51.25;
    param.SNOW_NEW_SNT_C3 = 2.59;
    param.SNOW_NEW_BRAS_DENOM = 100.;
    param.SNOW_MIN_SWQ_EB_THRES = 0.0010;
    param.SNOW_A1 = 0.7;
    param.SNOW_A2 = 0.3;
    param.SNOW_L1 = 6.0;
    param.SNOW_L2 = 20.0;
    param.SNOW_NEW_SNOW_ALB = 0.85;
    param.SNOW_ALB_ACCUM_A = 0.94;
    param.SNOW_ALB_ACCUM_B = 0.58;
    param.SNOW_ALB_THAW_A = 0.82;
    param.SNOW_ALB_THAW_B = 0.46;
    param.SNOW_TRACESNOW = 0.03;
    param.SNOW_CONDUCT = 2.9302e-6;
    param.SNOW_MAX_SNOW_TEMP = 0.5;
    param.SNOW_MIN_RAIN_TEMP = -0.5;

    // Blowing Snow Parameters
    param.BLOWING_KA = 0.0245187;
    param.BLOWING_CSALT = 0.68;
    param.BLOWING_UTHRESH = 0.25;
    param.BLOWING_KIN_VIS = 1.3e-5;
    param.BLOWING_MAX_ITER = 100;
    param.BLOWING_K = 5;
    param.BLOWING_SETTLING = 0.3;
    param.BLOWING_NUMINCS = 10;

    // Treeline temperature
    param.TREELINE_TEMPERATURE = 10.0;

    // Iteration bracket widths
    param.SNOW_DT = 5.0;
    param.SURF_DT = 1.0;
    param.SOIL_DT = 0.25;
    param.CANOPY_DT = 1.0;
    param.CANOPY_VP = 25.0;

    // Convergence Tolerances
    param.TOL_GRND = 0.001;
    param.TOL_OVER = 0.001;

    // Frozen Soil Parameters
    param.FROZEN_MAXITER = 1000;

    // Canopy Iterations
    // initialized to 10, set to 0 if
    // options.CLOSE_ENERGY is false
    // this allows for flexibility in
    // changing the maximum number of
    // iterations
    param.MAX_ITER_GRND_CANOPY = 10;

    // Newton-Raphson solver parameters
    param.NEWT_RAPH_MAXTRIAL = 150;
    param.NEWT_RAPH_TOLX = 1.0e-4;
    param.NEWT_RAPH_TOLF = 1.0e-1;
    param.NEWT_RAPH_R_MAX = 2.0;
    param.NEWT_RAPH_R_MIN = -5.0;
    param.NEWT_RAPH_RELAX1 = 0.9;
    param.NEWT_RAPH_RELAX2 = 0.7;
    param.NEWT_RAPH_RELAX3 = 0.2;
    param.NEWT_RAPH_EPS2 = 1.0e-4;

    // Root-Brent parameters
    param.ROOT_BRENT_MAXTRIES = 5;
    param.ROOT_BRENT_MAXITER = 1000;
    param.ROOT_BRENT_TSTEP = 10;
    param.ROOT_BRENT_T = 1.0e-7;
}
