% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/routing.R
\name{Lohmann_UH}
\alias{Lohmann_UH}
\alias{Lohmann_conv}
\title{Simulate the streamflow discharge of the VIC model outputs by the routing
model of Lohmann et al.}
\usage{
Lohmann_UH(dir_file, soil_params, stn_x, stn_y, fract = NULL, veloc = 1.5,
  diffu = 800, uh_box = NULL, arcinfo = TRUE)

Lohmann_conv(runoff_table, uh, out_monthly = TRUE)
}
\arguments{
\item{dir_file}{Flow direction raster file of the basin (should be
in ArcGIS style ASCII format). This describes the flow direction from a
gridcell to
its downstream neighboring gridcells. Details see the works of
\href{http://www.ntsg.umt.edu/project/drt.php}{Wu et al. (2012)}.}

\item{soil_params}{Soil parameter of the VIC model (a input of
\code{\link{vic}}).}

\item{stn_x}{X coordinate of the site to generate streamflow that Usually
are the basin outlet or hydrologial stations.}

\item{stn_y}{Y coordinate of the site to generate streamflow.}

\item{fract}{Path of fraction file, determining the fraction of the area
of the gridcells that actually in the basin. Must be a ArcGIS style ASCII
raster file that entirely
corresponding to \code{dir_file}. If not provided, those fractions would
be set to 1 (presume that all the gridcells of the routing model are
entirely located in the basin.}

\item{veloc}{velocity kinematic wave [m/s] of the river channel. Must be
the file path of a ArcGIS style ASCII raster file that entirely
corresponding to \code{dir_file},
or a single value for all gridcells. Default 1.5.}

\item{diffu}{Diffusion parameter [m^2/s] of the river channel. Must be the
file path of a ArcGIS style ASCII raster file that entirely corresponding
to \code{dir_file}, or a single value for all gridcells. Default 800.}

\item{uh_box}{Unit hydrograph describe the runoff from generated in the
gridcell to the river channel pass throught the gridcell. If not assigned,
a default unit hydrograph would be used.}

\item{arcinfo}{If the flow direction file using the direction code of ArcGIS
or not (See details). Default TRUE.}

\item{runoff_table}{Output table of the VIC model that containing variables:
`OUT_RUNOFF` and `OUT_BASEFLOW` in daily scale.}

\item{uh}{Output of the function \code{Lohmann_UH} that containing the unit
hydrograph informations of the basin.}

\item{out_monthly}{If is TRUE, it would also output the streamflow series
at monthly scale, else only outputs at daily scale.}
}
\value{
The daiy routing results (Streamflow series) of the VIC model.
}
\description{
Generate the daily streamflow output at the basin outlet using
the VIC model runoff outputs by the routing model developed by Lohman et al
(1996, 1998). The model is usually used in two steps: firstly, generate the
unit hydrographs (UH) of each gridcell for the outlet of the basin or the
situation of a hydrological station (for comparing the simulations with
observations) of the VIC model (use \code{Lohmann_UH});
and secondly, generate the streamflow using the UH inputed
by the VIC model runoff and baseflow outputs of each gridcell (use
`Lohmann_conv()`).
}
\details{
The streamflow routing of VIC model usually need two steps:
Step I build the unit hydrograph data of the basin, which needs information
of river channel networks of the basin (parameter \code{direc_file}),
coordinates of the gridcells of the VIC model (\code{soil_param}), location
of the pour point (parameter \code{stn_x} and \code{stn_y}, usually is the
location of hydrological station providing streamflow observations for
comparing), and some other information (wave velocity, diffusion, and the
unit hydrograph of runoff from land to river channel of each gridcell, i.e.
\code{uh_box}). This uses the function \code{Lohmann_UH()}:

\preformatted{
uh <- Lohmann_UH(direc_file, soil_params, stn_x, stn_y)
}

Step II generate the streamflow by using the unit hydrograph data of step I
and the runoff data output by the VIC model, using
the function \code{Lohmann_conv()}:

\preformatted{
rf <- Lohmann_conv(runoff_table, uh)
}

Where \code{rf} are the routed streamflow series.

The finer raster of routing model than VIC gridcells is supported. For
example, the routing model can be run at 0.125 degree spatial scale when
the VIC model is run at 0.25 degree scale.

The flow direction raster file should be a ASCII file in ArcGIS style like
that:

\preformatted{
ncols         5
nrows         4
xllcorner     -121.125
yllcorner     48.125
cellsize      0.125
NODATA_value  -1
-1   2     4     4    -1
1    1     2     16   8
1    128   1     0    16
-1   64    128   64   -1
}

The raster values are the direction codes. The fraction file, wave velocity
file and diffusion file should also in this
form and the rasters should be entirely corresponding to direc file (number
of rows and columns, size of gridcell, coordinates should be the same).

Direction code determines that the river channel would flow from a gridcell
to which one of the 8 gridcells surround the center gridcell.
Direction codes of ArcGis style:

\tabular{ccc}{
 32 \tab 64 \tab 128 \cr
 16 \tab 0  \tab  1  \cr
 8  \tab 4  \tab  2
}

Direction codes of not ArcGIS:

\tabular{ccc}{
 8 \tab 1 \tab 2 \cr
 7 \tab 0 \tab 3 \cr
 6 \tab 5 \tab 4
}

The fraction file (\code{fract}) determines the fraction of the gridcells
that located in the realistic basin. The inner gridcells that entirely located in
the basin should be with the value 1, while for the outer gridcells, with
the boundary pass through, would have a part of runoff that flow into the
neighbouring basin and therefore those part of the runoff would not be
calculated in the streamflow routing. For the cases with large number of
gridcells, those effects could be ignore and could not provide the fraction
file.

An example of the fraction file:

\preformatted{
ncols         5
nrows         4
xllcorner     -121.125
yllcorner     48.125
cellsize      0.125
NODATA_value  -1
0.000 0.147 0.231 0.173 0.000
0.320 1.000 1.000 0.916 0.053
0.213 0.978 1.000 0.738 0.169
0.000 0.213 0.084 0.049 0.000
}

\code{runoff_table} should be a output table of function \code{\link{vic}}
that containing two variables: \code{OUT_RUNOFF} and \code{OUT_BASEFLOW}.
Thus the parameter \code{output_info} of \code{\link{vic}} can be set as:

\preformatted{
out_info <- list(
  runoff_table = list(
            timescale = 'day', aggpar = 1,
            outvars = c('OUT_RUNOFF', 'OUT_BASEFLOW'),
            aggtypes = c('sum', 'sum')
            )
)

res <- vic(forcing, soil, veg, output_info = out_info)
}

And then run the streamflow routing as:

\preformatted{
sf <- Lohmann_conv(res$runoff_table, uh)
}
}
\examples{
# Paths of the samples of the flow direction file and fraction file
direc_file <- system.file("extdata", "direc_STEHE.asc", package = "VICmodel")
fract_file <- system.file("extdata", "fract_STEHE.asc", package = "VICmodel")

# Generate the unit hydrograph data of each gridcells.
uh <- Lohmann_UH(direc_file, STEHE$soil, stn_x = -120.7, stn_y = 48.31,
                 fract = fract_file)

# Streamflow routing using the VIC output
sf <- Lohmann_conv(STEHE$runoff_table_daily, uh)

# Draw the output hydrograph
plot(sf$daily, type = 'l')

obs <- STEHE$streamflow_obs
plot(obs, type = 'l')
lines(sf$monthly, col = 'blue')

}
\references{
Lohmann D, Nolte-Holube R, Raschke E, 1996. A large-scale
horizontal routing model to be coupled to land surface parametrization
schemes. Tellus A, 48(5): 708-721.

Lohmann D, Raschke E, Nijssen B, et al., 1998. Regional scale hydrology: I.
Formulation of the VIC-2L model coupled to a routing model. Hydrological
Sciences Journal, 43(1): 131-141.

Wu H, Kimball JS, Li H, Huang M, Leung LR, and Adler RF, 2012. A new global
river network database for macroscale hydrologic modeling.Water Resources
Research, 48, W09701.
}

