% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abc_grid.R
\name{grid_search}
\alias{grid_search}
\title{Grid Search}
\usage{
grid_search(x, y, range_gamma, range_cost, method = "estimator",
  nfolds = 10, bias_correction = TRUE)
}
\arguments{
\item{x}{Input matrix or data.frame of dimension \code{nobs x nvars}; each row is an feature vector.}

\item{y}{A numeric vector or factor of class labels. Factor should have either two levels or be
a vector with two distinct values.
If \code{y} is presented as a vector, it will be coerced into a factor.
Length of \code{y} has to correspond to number of samples in \code{x}.}

\item{range_gamma}{Vector of \code{gamma} values to check.}

\item{range_cost}{nobs x 1 vector (values should be between 0 and 1) or
nobs x 2 matrix (each row is cost pair value c(\eqn{C_{10}}{C_10}, \eqn{C_{01}}{C_01}))
of cost values to check.}

\item{method}{Selects method to evaluete risk. "estimator" and "cross".}

\item{nfolds}{Number of folds to use with cross-validation. Default is 10.
In case of imbalanced data, \code{nfolds} should not be greater than the number of observations in
smaller class.}

\item{bias_correction}{Takes in a boolean value.
If \code{bias_correction} is TRUE, then asymptotic bias correction will be performed.
Otherwise, (if \code{bias_correction} is FALSE) asymptotic bias correction will not be performed and
the ABCRLDA is the classical RLDA.
The default is TRUE.}
}
\value{
List of estimated parameters.
  \item{cost}{Cost value for which risk estimates are lowest during the search.}
  \item{gamma}{Gamma regularization parameter for which risk estimates are lowest during the search.}
  \item{risk}{Lowest risk value estimated during grid search.}
}
\description{
Performs grid search to estimate the optimal hyperparameters (\code{gamma} and \code{cost})
  within specified space based on double asymptotic risk estimation or cross validation.
  Double asymptotic risk estimation is more efficient to compute because it uses closed form for risk estimation.
  For further details, refer to the article in the reference section.
  \deqn{\Re = \varepsilon_0 * C_{10} + \varepsilon_1 * C_{01}}{R = e_0 * C_10 + e_1 * C_01)}
  \deqn{\varepsilon_i = \Phi(\frac{(-1)^{i+1} ( \hat{G}_i + \hat{\omega}_{opt}/\gamma   )}{\sqrt{\hat{D}}})}{e_i = CDF((-1)^(i+1) (Ghat_i + omega_opt/gamma) / sqrt(Dhat))}
  Separate sampling cross-validation (see cross-validation function) was adapted to work with cost-based risk estimation.
}
\section{Reference}{

  A. Zollanvari, M. Abdirash, A. Dadlani and B. Abibullaev,
  "Asymptotically Bias-Corrected Regularized Linear Discriminant Analysis for Cost-Sensitive
  Binary Classification," in IEEE Signal Processing Letters, vol. 26, no. 9, pp. 1300-1304,
  Sept. 2019. doi: 10.1109/LSP.2019.2918485
  URL: \url{http://ieeexplore.ieee.org/stamp/stamp.jsp?tp=&arnumber=8720003&isnumber=8770167}

  Braga-Neto, Ulisses & Zollanvari, Amin & Dougherty, Edward. (2014).
  Cross-Validation Under Separate Sampling: Strong Bias and How to Correct It.
  Bioinformatics (Oxford, England). 30. 10.1093/bioinformatics/btu527.
  URL: \url{https://www.ncbi.nlm.nih.gov/pmc/articles/PMC4296143/pdf/btu527.pdf}
}

\examples{
data(iris)
train_data <- iris[which(iris[, ncol(iris)] == "virginica" |
                         iris[, ncol(iris)] == "versicolor"), 1:4]
train_label <- factor(iris[which(iris[, ncol(iris)] == "virginica" |
                                 iris[, ncol(iris)] == "versicolor"), 5])
cost_range <- seq(0.1, 0.9, by = 0.2)
gamma_range <- c(0.1, 1, 10, 100, 1000)

gs <- grid_search(train_data, train_label,
                  range_gamma = gamma_range,
                  range_cost = cost_range,
                  method = "estimator")
model <- abcrlda(train_data, train_label,
                 gamma = gs$gamma, cost = gs$cost)
predict(model, train_data)

cost_range <- matrix(1:10, ncol = 2)
gamma_range <- c(0.1, 1, 10, 100, 1000)

gs <- grid_search(train_data, train_label,
                  range_gamma = gamma_range,
                  range_cost = cost_range,
                  method = "cross")
model <- abcrlda(train_data, train_label,
                 gamma = gs$gamma, cost = gs$cost)
predict(model, train_data)
}
\seealso{
Other functions in the package: \code{\link{abcrlda}},
  \code{\link{cross_validation}},
  \code{\link{da_risk_estimator}},
  \code{\link{predict.abcrlda}},
  \code{\link{risk_calculate}}
}
\concept{functions in the package}
