% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/04-aghq.R
\name{aghq}
\alias{aghq}
\title{Adaptive Gauss-Hermite Quadrature}
\usage{
aghq(ff, k, startingvalue, optresults = NULL, control = default_control(), ...)
}
\arguments{
\item{ff}{A list with three elements:
\itemize{
\item{\code{fn}}{: function taking argument \code{theta} and returning a numeric
value representing the log-posterior at \code{theta}}
\item{\code{gr}}{: function taking argument \code{theta} and returning a numeric
vector representing the gradient of the log-posterior at \code{theta}}
\item{\code{he}}{: function taking argument \code{theta} and returning a numeric
matrix representing the hessian of the log-posterior at \code{theta}}
}
The user may wish to use \code{numDeriv::grad} and/or \code{numDeriv::hessian} to
obtain these. Alternatively, the user may consider the \code{TMB} package. This
list is deliberately formatted to match the output of \code{TMB::MakeADFun}.}

\item{k}{Integer, the number of quadrature points to use. I suggest at least 3. k = 1 corresponds to a Laplace
approximation.}

\item{startingvalue}{Value to start the optimization. \code{ff$fn(startingvalue)},
\code{ff$gr(startingvalue)}, and \code{ff$he(startingvalue)} must all return
appropriate values without error.}

\item{optresults}{Optional. A list of the results of the optimization of the log
posterior, formatted according to the output of \code{aghq::optimize_theta}. The
\code{aghq::aghq} function handles the optimization for you; passing this list
overrides this, and is useful for when you know your optimization is too difficult to be
handled by general-purpose software. See the software paper for several examples of this.
If you're unsure whether this option is needed for your problem then it probably is not.}

\item{control}{A list with elements
\itemize{
\item{\code{method}: }{optimization method to use:
\itemize{
\item{'sparse_trust' (default): }{\code{trustOptim::trust.optim} with \code{method = 'sparse'}}
\item{'SR1' (default): }{\code{trustOptim::trust.optim} with \code{method = 'SR1'}}
\item{'trust': }{\code{trust::trust}}
\item{'BFGS': }{\code{optim(...,method = "BFGS")}}
}
Default is 'sparse_trust'.
}
\item{\code{optcontrol}: }{optional: a list of control parameters to pass to the
internal optimizer you chose. The \code{aghq} package uses sensible defaults.}
}}

\item{...}{Additional arguments to be passed to \code{ff$fn}, \code{ff$gr}, and \code{ff$he}.}
}
\value{
An object of class \code{aghq} which is a list containing elements:
\itemize{
\item{normalized_posterior: }{The output of the \code{normalize_logpost} function, which
itself is a list with elements:
\itemize{
\item{\code{nodesandweights}: }{a dataframe containing the nodes and weights for the adaptive quadrature rule, with the un-normalized and normalized log posterior evaluated at the nodes.}
\item{\code{thegrid}: }{a \code{NIGrid} object from the \code{mvQuad} package, see \code{?mvQuad::createNIGrid}.}
\item{\code{lognormconst}: }{the actual result of the quadrature: the log of the normalizing constant of the posterior.}
}}
\item{marginals: }{a list of the same length as \code{startingvalue} of which element \code{j}
is the result of calling \code{aghq::marginal_posterior} with that \code{j}. This is
a tbl_df/tbl/data.frame containing the normalized log marginal posterior
for theta_j evaluated at the original quadrature points. Has columns
\code{"thetaj","logpost_normalized","weights"}, where \code{j} is the \code{j} you specified.
}
\item{optresults: }{information and results from the optimization of the log posterior, the
result of calling \code{aghq::optimize_theta}. This a list with elements:
\itemize{
\item{\code{ff}: }{the function list that was provided}
\item{\code{mode}: }{the mode of the log posterior}
\item{\code{hessian}: }{the hessian of the log posterior at the mode}
\item{\code{convergence}: }{specific to the optimizer used, a message indicating whether it converged}
}
}
}
}
\description{
Normalize the log-posterior distribution using Adaptive Gauss-Hermite Quadrature.
This function takes in a function and its gradient and Hessian, and returns
a list of information about the normalized posterior, with methods for summarizing
and plotting.
}
\details{
When \code{k = 1} the AGHQ method is a Laplace approximation, and you should use
the \code{aghq::laplace_approximation} function, since some of the methods for
\code{aghq} objects won't work with only one quadrature point. Objects of
class \code{laplace} have different methods suited to this case. See \code{?aghq::laplace_approximation}.
}
\examples{

logfteta2d <- function(eta,y) {
  # eta is now (eta1,eta2)
  # y is now (y1,y2)
  n <- length(y)
  n1 <- ceiling(n/2)
  n2 <- floor(n/2)
  y1 <- y[1:n1]
  y2 <- y[(n1+1):(n1+n2)]
  eta1 <- eta[1]
  eta2 <- eta[2]
  sum(y1) * eta1 - (length(y1) + 1) * exp(eta1) - sum(lgamma(y1+1)) + eta1 +
    sum(y2) * eta2 - (length(y2) + 1) * exp(eta2) - sum(lgamma(y2+1)) + eta2
}
set.seed(84343124)
n1 <- 5
n2 <- 5
n <- n1+n2
y1 <- rpois(n1,5)
y2 <- rpois(n2,5)
objfunc2d <- function(x) logfteta2d(x,c(y1,y2))
funlist2d <- list(
  fn = objfunc2d,
  gr = function(x) numDeriv::grad(objfunc2d,x),
  he = function(x) numDeriv::hessian(objfunc2d,x)
)

thequadrature <- aghq(funlist2d,3,c(0,0))

}
\seealso{
Other quadrature: 
\code{\link{laplace_approximation}()},
\code{\link{marginal_laplace}()},
\code{\link{normalize_logpost}()},
\code{\link{optimize_theta}()},
\code{\link{plot.aghq}()},
\code{\link{print.aghqsummary}()},
\code{\link{print.aghq}()},
\code{\link{print.laplacesummary}()},
\code{\link{print.laplace}()},
\code{\link{summary.aghq}()},
\code{\link{summary.laplace}()}
}
\concept{quadrature}
