\name{gge}
\alias{gge}
\alias{gge.formula}
\alias{gge.matrix}
\alias{biplot.gge}
\alias{plot.gge}

\title{
  GGE biplots
}
\description{
  Fit a GGE (genotype + genotype * environment) model and
  display the results.
}
\usage{
gge(x, ...)

\method{gge}{formula}(formula, data = NULL, gen.group = NULL, env.group = NULL, ...)

\method{gge}{matrix}(x, center = TRUE, scale = TRUE, gen.group=NULL,
  env.group = NULL, comps=c(1,2), method = "svd", ...)

\method{plot}{gge}(x, title = substitute(x), ...)

\method{biplot}{gge}(x, title = substitute(x), subtitle = "",
  cex.gen = 0.6, cex.env = 0.5,
  col.gen = "darkgreen", col.env="orange3",
  pch.gen=1, lab.env = TRUE, comps = 1:2, flip = "auto", res.vec=TRUE, ...)

}

\arguments{
  \item{x}{A matrix or data.frame.}
  \item{formula}{A formula}
  \item{data}{Data frame}
  \item{\dots}{Other arguments}
  \item{title}{title}
  \item{center}{If TRUE, center values for each environment}
  \item{scale}{If TRUE, scale values for each environment}
  \item{subtitle}{subtitle}
  \item{gen.group}{genotype group}
  \item{env.group}{env group}
  \item{cex.gen}{character expansion for genotypes}
  \item{cex.env}{character expansion for environments}
  \item{col.gen}{color for genotypes}
  \item{col.env}{color for envts}
  \item{pch.gen}{plot character for genotypes}
  \item{lab.env}{label envts}
  \item{comps}{comps}
  \item{flip}{If "auto" then each axis is flipped so that the genotype
    ordinate is positively correlated with genotype means.  Can also be
    a vector like c(TRUE,FALSE) for manual control.}
  \item{res.vec}{If TRUE, for each group, draw residual vectors from the
    mean of the locs to the individual locs}
  \item{method}{method used to find principal component directions}
}
\details{
  If there is replication in G*E, then the replications are
  averaged together before constructing the biplot.

  The singular value decomposition of \code{x} is used to
  calculate the principal components for the biplot.  Missing
  values are NOT allowed.

  The NIPALS algorithm can be used when there are missing data.

  The argument 'method' can be either 'svd' for complete-data, or
  'nipals' for missing-data.

}
\value{
  A list of class \code{gge} containing:
  \item{method}{}
  \item{center}{Data centered?}
  \item{scale}{Data scaled?}
  \item{gen.group}{
    This is only used for plotting.
    If not NULL, this specifies a classification of genotypes into
    groups.
  }
  \item{env.group}{
    If not NULL, this specifies a classification of
    environments into groups.  When using the formula method, then
    \code{envGroup} is the name of a column in the \code{data}.
    When using the matrix method, then \code{envGroup} must be a vector
    the same length as the number of columns in \code{data}, the
    contents of the vector contain the grouping information.
  }

}
\references{
  Jean-Louis Laffont, Kevin Wright and Mohamed Hanafi (2013).
  Genotype + Genotype x Block of Environments (GGB) Biplots.
  \emph{Crop Science}, 53, 2332-2341.
  \url{https://www.crops.org/publications/cs/abstracts/53/6/2332}.

  Kroonenberg, Pieter M. (1997) \emph{Introduction to Biplots
    for GxE Tables}, Research Report 51, Centre for Statistics,
  The University of Queensland, Brisbane, Australia.
  \url{http://three-mode.leidenuniv.nl/document/biplot.pdf}

  Yan, W. and Kang, M.S. (2003) \emph{GGE Biplot Analysis}.
  CRC Press.

}
\author{
  Jean-Louis Laffont, Kevin Wright
}

\seealso{
  The 'agricolae' package has functions for AMMI biplots.
}
\examples{

# Example 1.  Data is a data.frame in 'matrix' format
B <- matrix(c(50, 67, 90, 98, 120,
              55, 71, 93, 102, 129,
              65, 76, 95, 105, 134,
              50, 80, 102, 130, 138,
              60, 82, 97, 135, 151,
              65, 89, 106, 137, 153,
              75, 95, 117, 133, 155), ncol=5, byrow=TRUE)
rownames(B) <- c("G1","G2","G3","G4","G5","G6","G7")
colnames(B) <- c("E1","E2","E3","E4","E5")

m1 = gge(B)
plot(m1)
biplot(m1, title="Example biplot")

# crossa.wheat biplot

# Specify env.group as column in data frame
data(crossa.wheat)
dat2 <- crossa.wheat
dat2$eg <- ifelse(is.element(dat2$loc,
c("KN","NB","PA","BJ","IL","TC", "JM","PI","AS","ID","SC","SS",
"SJ","MS","MG","MM")), "Grp1", "Grp2")
m4 <- gge(yield~gen*loc, dat2, env.group=eg, scale=FALSE)
biplot(m4, lab.env=TRUE, title="crossa.wheat")

}

\keyword{hplot}

