# Copyright © 2016 RTE Réseau de transport d’électricité

#' plot time series contained in an antaresData object
#' 
#' This function generates an interactive plot of an antares time series.
#' 
#' @param x
#'   Object of class \code{antaresData}.
#' @param y
#'   Optional object of class \code{antaresData}. If it is specified, then two
#'   charts are generated.
#' @param table
#'   Name of the table to display when \code{x} is an \code{antaresDataList}
#'   object.
#' @param variable
#'   Name of the variable to plot. If this argument is missing, then the 
#'   function starts a shiny gadget that let the user choose the variable to
#'   represent. When the user clicks on the "Done" button", the graphic is
#'   returned by the function.
#' @param elements
#'   Vector of "element" names indicating for which elements of 'x' should the
#'   variable be plotted. For instance if the input data contains areas, then
#'   this parameter should be a vector of area names. If data contains clusters
#'   data, this parameter has to be the concatenation of the area name and the
#'   cluster name, separated by \code{" > "}. This is to prevent confusion 
#'   when two clusters from different areas have the same name.
#' @param type
#'   Type of plot to draw. "ts" creates a time series plot, "barplot" creates
#'   a barplot with one bar per element representing the average value of the
#'   variable for this element. "monotone" draws the monotone curve of the 
#'   variable for each element.
#' @param dateRange
#'   A vector of two dates. Only data points between these two dates are 
#'   displayed. If NULL, then all data is displayed.
#' @param confInt
#'   Number between 0 and 1 indicating the size of the confidence interval to 
#'   display. If it equals to 0, then confidence interval is not computed nor
#'   displayed. Used only when multiple Monte Carlo scenarios are present in 
#'   the input data.
#' @param minValue
#'   Only used if parameter \code{type} is "density" or "cdf". If this parameter
#'   is set, all values that are less than \code{minValue} are removed from the 
#'   graphic. This is useful to deal with variables containing a few extreme
#'   values (generally cost and price variables). If \code{minValue} is unset,
#'   all values are displayed.
#' @param maxValue
#'   Only used if parameter \code{type} is "density" or "cdf". If this parameter 
#'   is set, all values not in [-minValue, maxValue] are removed from the graphic.
#'   This is useful to deal with variables containing a few extreme values
#'   (generally cost and price variables). If \code{maxValue} is 0 or unset, all
#'   values are displayed.
#' @param aggregate
#'   When multiple elements are selected, should the data be aggregated. If
#'   "none", each element is represented separetly. If "mean" values are
#'   averaged and if "sum" they are added.
#' @param colors
#'   Vector of colors
#' @param ylab
#'   Label of the Y axis.
#' @param compare
#'   An optional character vector containing names of parameters. When it is set,
#'   two charts are outputed with their own input controls. Alternatively, it can
#'   be a named list with names corresponding to parameter names and values being
#'   list with the initial values of the given parameter for each chart.
#' @param compareLayout
#'   Only used if \code{y} or \code{compare} is not null. If it equals to "v", 
#'   then charts are placed one above the other. If it equals to "h", they are
#'   placed one next to the other.
#' @param colorScaleOpts
#'   A list of parameters that control the creation of color scales. It is used
#'   only for heatmaps. See \code{\link{colorScaleOptions}}() for available
#'   parameters.
#'   
#' @param ...
#'   currently unused
#' @inheritParams prodStack
#'   
#' @return 
#' The function returns an object of class "htmlwidget". It is generated by
#' package \code{highcharter} if time step is annual or by \code{dygraphs} for 
#' any other time step.It can be directly displayed in the viewer or be stored
#' in a variable for later use.
#' 
#' @details 
#' If the input data contains several Monte-Carlo scenarios, the function will
#' display the evolution of the average value. Moreover it will represent a
#' 95% confidence interval.
#' 
#' If the input data has a annual time step, the function creates a barplot
#' instead of a line chart.
#' 
#' 
#' @examples 
#' \dontrun{
#' setSimulationPath()
#' mydata <- readAntares("all", timeStep = "monthly")
#' plot(mydata)
#' plot(mydata, "LOAD")
#' 
#' # Plot only a few areas
#' plot(mydata[area %in% c("area1", "area2", "area3")])
#' 
#' # If data contains detailed results, then the function adds a confidence
#' # interval
#' dataDetailed <- readAntares("all", timeStep = "monthly", synthesis = FALSE)
#' plot(dataDetailed)
#' 
#' # If the time step is annual, the function creates a barplot instead of a
#' # linechart
#' dataAnnual <- readAntares("all", timeStep = "Annual")
#' plot(dataAnnual)
#' 
#' # Compare the results of two simulations
#' setSimulationPath(path1)
#' mydata1 <- readAntares("all", timeStep = "daily")
#' setSimulationPath(path2)
#' mydata2 <- readAntares("all", timeStep = "daily")
#' 
#' plot(mydata1, mydata2)
#' 
#' # Compare two periods for the same simulation
#' plot(mydata1, compare = "dateRange")
#' 
#' # Compare two Monte-Carlo scenarios
#' detailedData <- readAntares("all", mcYears = "all")
#' plot(detailedData[mcYear == 1], detailedData[mcYear == 2])
#' 
#' # To do the same thing, with antaresDataList objects, one can use 'subset'
#' detailedData <- readAntares(areas = "all" links = "all", mcYears = "all")
#' plot(subset(detailedData, mcYears = 1), subset(detailedData, mcYears = 2))
#' }
#' 
#' @export
plot.antaresData <- function(x, y = NULL, table = NULL, variable = NULL, elements = NULL, 
                             mcYear = "average",
                             type = c("ts", "barplot", "monotone", "density", "cdf", "heatmap"),
                             dateRange = NULL,
                             confInt = 0,
                             minValue = NULL,
                             maxValue = NULL,
                             aggregate = c("none", "mean", "sum"),
                             compare = NULL,
                             compareLayout = c("v", "h"),
                             interactive = getInteractivity(),
                             colors = NULL,
                             main = NULL,
                             ylab = NULL,
                             legend = TRUE,
                             legendItemsPerRow = 5,
                             colorScaleOpts = colorScaleOptions(20),
                             width = NULL, height = NULL, ...) {
  
  dataname <- deparse(substitute(x))
  type <- match.arg(type)
  aggregate <- match.arg(aggregate)
  colorScaleOpts <- do.call(colorScaleOptions, colorScaleOpts)
  
  # Generate a group number for dygraph objects
  if (!("dateRange" %in% compare)) {
    group <- sample(1e9, 1)
  } else {
    group <- NULL
  }
  
  x <- as.antaresDataList(x)
  
  timeStep <- attr(x, "timeStep")
  compareLayout <- match.arg(compareLayout)
  opts <- simOptions(x)
  
  if (is.null(compare)) {
    if (!is.null(y)) compare <- list()
  } else {
    if (is.character(compare)) {
      compare <- match.arg(
        compare, 
        c("table", "variable", "elements", "type", "dateRange", "minValue", "maxValue", "mcYear"),
        several.ok = TRUE
      )
      tmp <- lapply(compare, function(x) NULL)
      names(tmp) <- compare
      compare <- tmp
    }
  }
  
  
  .prepareParams <- function(x) {
    idCols <- .idCols(x)
    
    dt <- x[, .(
      timeId = timeId,
      time = .timeIdToDate(timeId, timeStep, simOptions(x)), 
      value = 0)
    ]
    
    if ("cluster" %in% idCols) {
      dt$element <- paste(x$area, x$cluster, sep = " > ")
    } else if ("district" %in% idCols) {
      dt$element <- x$district
    } else if ("link" %in% idCols) {
      dt$element <- x$link
    } else if ("area" %in% idCols) {
      dt$element <- x$area
    } else stop("No Id column")
    
    if ("mcYear" %in% names(x) && length(unique(x$mcYear)) > 1) {
      dt$mcYear <- x$mcYear
    }
    
    dataDateRange <- as.Date(range(dt$time))
    if (is.null(dateRange) || length(dateRange) < 2) dateRange <- dataDateRange
    
    uniqueElem <- sort(as.character(unique(dt$element)))
    if (is.null(elements)) {
      elements <- uniqueElem
      if (length(elements) > 5) elements <- elements[1:5]
    }
    
    list(
      dt = dt,
      idCols = idCols,
      valueCols = setdiff(names(x), idCols),
      showConfInt = !is.null(x$mcYear) && length(unique(x$mcYear) > 1),
      dataDateRange = dataDateRange,
      dateRange = dateRange,
      uniqueElem = uniqueElem,
      uniqueMcYears = unique(x$mcYear),
      elements = elements
    )
  }
    
  params <- list(
    x = lapply(x, .prepareParams)
  )
  
  if (!is.null(y)) {
    if (!is(y, "antaresData")) stop("y should be an 'antaresData' object")
    y <- as.antaresDataList(y)
    params$y <- lapply(y, .prepareParams)
    x <- list(x, y)
  } else {
    params$y <- params$x
    x <- list(x, x)
  }
  
  # Function that generates the desired graphic.
  plotFun <- function(table, mcYear, id, variable, elements, type, confInt, dateRange, 
                      minValue, maxValue, aggregate) {
    
    if (is.null(variable)) variable <- params[[id]][[table]]$valueCols[1]
    if (is.null(dateRange)) dateRange <- params[[id]][[table]]$dateRange
    if (is.null(type) || is.null(table) || !variable %in% names(x[[id]][[table]])) {
      return(combineWidgets())
    }
    if (length(elements) == 0) {
      return(combineWidgets("Choose at least one element"))
    }
    
    dt <- params[[id]][[table]]$dt
    dt$value <- x[[id]][[table]][, get(variable)]
    
    if (!is.null(mcYear) && mcYear != "average") {
      mcy <- mcYear
      dt <- dt[mcYear == mcy]
    }
    
    if (length(elements) == 0) {
      elements <- params[[id]][[table]]$uniqueElem[1:5]
    }
    if (!"all" %in% elements) dt <- dt[element %in% elements]
    dt <- dt[as.Date(time) %between% dateRange]
    
    if (nrow(dt) == 0) return(combineWidgets())
    
    if (aggregate != "none" && length(params[[id]][[table]]$uniqueElem) > 1) {
      if (aggregate == "mean") {
        dt <- dt[, .(element = as.factor(variable), value = mean(value)), 
                 by = c(.idCols(dt))]
      } else if (aggregate == "sum") {
        dt <- dt[, .(element = as.factor(variable), value = sum(value)), 
                 by = c(.idCols(dt))]
      }
    }
    
    f <- switch(type,
                "ts" = .plotTS,
                "barplot" = .barplot,
                "monotone" = .plotMonotone,
                "density" = .density,
                "cdf" = .cdf,
                "heatmap" = .heatmap,
                stop("Invalid type")
    )
    f(
      dt, 
      timeStep = timeStep, 
      variable = variable, 
      confInt = confInt, 
      minValue = minValue,
      maxValue = maxValue, 
      colors = colors, 
      main = if(length(main) == 1) main else main[id], 
      ylab = if(length(ylab) == 1) ylab else ylab[id], 
      legend = legend, 
      legendItemsPerRow = legendItemsPerRow, 
      width = width, 
      height = height,
      opts = opts,
      colorScaleOpts = colorScaleOpts,
      group = group
    )
    
  }
  
  if (is.null(table)) table <- names(params[[1]])[1]
  if (is.null(mcYear)) mcYear <- "average"
  # If not in interactive mode, generate a simple graphic, else create a GUI
  # to interactively explore the data
  if (!interactive) {
    return(plotFun(table, mcYear, 1, variable, elements, type, confInt, dateRange, 
                   minValue, maxValue, aggregate))
  }
  
  typeChoices <- c("time series" = "ts", "barplot", "monotone", "density", "cdf", "heatmap")
  
  manipulateWidget(
    plotFun(table, mcYear, .id, variable, elements, type, confInt, dateRange, minValue, 
            maxValue, aggregate),
    
    table = mwSelect(names(params[[.id]]), value = table, .display = length(params[[.id]]) > 1),
    mcYear = mwSelect(
      choices = c("average", params[[.id]][[table]]$uniqueMcYears) ,
      mcYear, 
      .display = params[[.id]][[table]]$showConfInt
    ),
    variable = mwSelect(
      choices = params[[.id]][[table]]$valueCols,
      value = variable
    ),
    type = mwSelect(
      choices = {
        if (timeStep == "annual") "barplot"
        else if (timeStep %in% c("hourly", "daily")) typeChoices
        else typeChoices[1:5]
      },
      value = type, 
      .display = timeStep != "annual"
    ),
    dateRange = mwDateRange(
      value = params[[1]][[table]]$dateRange,
      min = params[[.id]][[table]]$dataDateRange[1], 
      max = params[[.id]][[table]]$dataDateRange[2],
      .display = timeStep != "annual"
    ),
    confInt = mwSlider(0, 1, confInt, step = 0.01, label = "confidence interval",
                       .display = params[[.id]][[table]]$showConfInt & mcYear == "average"),
    minValue = mwNumeric(minValue, "min value", .display = type %in% c("density", "cdf")),
    maxValue = mwNumeric(maxValue, "max value", .display = type %in% c("density", "cdf")),
    elements = mwSelect(
      choices = c("all", params[[.id]][[table]]$uniqueElem),
      value = elements, 
      multiple = TRUE
    ),
    aggregate = mwSelect(c("none", "mean", "sum"), aggregate),
    
    .main = dataname,
    .compare = compare,
    .compareLayout = compareLayout
  )
  
}
