% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_control.R
\name{orsf_control_custom}
\alias{orsf_control_custom}
\title{Custom ORSF control}
\usage{
orsf_control_custom(beta_fun, ...)
}
\arguments{
\item{beta_fun}{(\emph{function}) a function to define coefficients used
in linear combinations of predictor variables. \code{beta_fun} must accept
three inputs named \code{x_node}, \code{y_node} and \code{w_node}, and should expect
the following types and dimensions:
\itemize{
\item \code{x_node} (\emph{matrix}; \code{n} rows, \code{p} columns)
\item \code{y_node} (\emph{matrix}; \code{n} rows, \code{2} columns)
\item \code{w_node} (\emph{matrix}; \code{n} rows, \code{1} column)
}

In addition, \code{beta_fun} must return a matrix with p rows and 1 column. If
any of these conditions are not met, \code{orsf_control_custom()} will let
you know.}

\item{...}{Further arguments passed to or from other methods (not currently used).}
}
\value{
an object of class \code{'orsf_control'}, which should be used as
an input for the \code{control} argument of \link{orsf}.
}
\description{
Custom ORSF control
}
\section{Examples}{
Two customized functions to identify linear combinations of predictors
are shown here.
\itemize{
\item The first uses random coefficients
\item The second derives coefficients from principal component analysis.
}
\subsection{Random coefficients}{

\code{f_rando()} is our function to get the random coefficients:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_rando <- function(x_node, y_node, w_node)\{
 matrix(runif(ncol(x_node)), ncol=1) 
\}
}\if{html}{\out{</div>}}

We can plug \code{f_rando} into \code{orsf_control_custom()}, and then pass the
result into \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(aorsf)

fit_rando <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  control = orsf_control_custom(beta_fun = f_rando),
                  n_tree = 500)

fit_rando
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Custom user function
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 20
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}
}

\subsection{Principal components}{

Follow the same steps as above, starting with the custom function:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_pca <- function(x_node, y_node, w_node) \{ 
 
 # estimate two principal components.
 pca <- stats::prcomp(x_node, rank. = 2)
 # use the second principal component to split the node
 pca$rotation[, 2L, drop = FALSE]
 
\}
}\if{html}{\out{</div>}}

Then plug the function into \code{orsf_control_custom()} and pass the result
into \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_pca <- orsf(pbc_orsf,
                Surv(time, status) ~ . - id,
                control = orsf_control_custom(beta_fun = f_pca),
                n_tree = 500)
}\if{html}{\out{</div>}}
}

\subsection{Evaluate}{

How well do our two customized ORSFs do? Let’s compute their indices of
prediction accuracy based on out-of-bag predictions:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(riskRegression)
library(survival)

risk_preds <- list(rando = 1 - fit_rando$pred_oobag,
                    pca = 1 - fit_pca$pred_oobag)

sc <- Score(object = risk_preds, 
            formula = Surv(time, status) ~ 1, 
            data = pbc_orsf, 
            summary = 'IPA',
            times = fit_pca$pred_horizon)
}\if{html}{\out{</div>}}

The PCA ORSF does quite well! (higher IPA is better)

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$Brier
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## 
## Results by model:
## 
##         model times  Brier  lower  upper    IPA
## 1: Null model  1788 20.479 18.090 22.868  0.000
## 2:      rando  1788 11.604  9.535 13.673 43.339
## 3:        pca  1788 12.870 10.872 14.869 37.154
## 
## Results of model comparisons:
## 
##    times model  reference delta.Brier   lower  upper            p
## 1:  1788 rando Null model      -8.875 -11.063 -6.688 1.852437e-15
## 2:  1788   pca Null model      -7.609  -9.351 -5.866 1.143284e-17
## 3:  1788   pca      rando       1.267   0.449  2.084 2.381056e-03

## 
## NOTE: Values are multiplied by 100 and given in \%.

## NOTE: The lower Brier the better, the higher IPA the better.
}\if{html}{\out{</div>}}
}
}

\seealso{
linear combination control functions
\code{\link{orsf_control_cph}()},
\code{\link{orsf_control_fast}()},
\code{\link{orsf_control_net}()}
}
\concept{orsf_control}
