#' Anytime-valid upper confidence limit for the probability parameter of a
#' sample from the binomial distribution, based on construction of Robbins
#' (1970).
#'
#' @param S number of successes
#' @param n number of trials
#' @param epsilon significance level of the confidence sequence
#'
#' @return Upper confidence limit
#' @keywords internal
#' @importFrom stats uniroot dbinom
#' @note Produces estimates at a precision of .Machine$double.eps^0.5. As a
#'   result, for very large values of n, inverting the result of this function
#'   via invert.ptilde may not accurately recover S.
Un_tilde <- function(S,n,epsilon){
  if(S/n < 1){
    return(min(1,uniroot(function(p){dbinom(S,n,p) - epsilon/(n+1)},c(S/n,1),
                         tol=.Machine$double.eps^0.5)$root))
  }else{
    return(1)
  }
}

#' Anytime-valid lower confidence limit for the probability parameter of a
#' sample from the binomial distribution, based on construction of Robbins
#' (1970).
#'
#' @param S number of successes
#' @param n number of trials
#' @param epsilon significance level of the confidence sequence
#'
#' @return Lower confidence limit
#' @keywords internal
#' @importFrom stats uniroot dbinom
#' @note Produces estimates at a precision of .Machine$double.eps^0.5. As a
#'   result, for very large values of n, inverting the result of this function
#'   via invert.ptilde may not accurately recover S.
Ln_tilde <- function(S,n,epsilon){
  if(S/n > 0){
    return(max(0, uniroot(function(p){dbinom(S,n,p) - epsilon/(n+1)},c(0,S/n),
                          tol=.Machine$double.eps^0.5)$root))
  }else{
    return(0)
  }
}

#' Finds the number of successes S based on the upper limit of the confidence
#' sequence vased on the construction by Robbins (1970) (e.g. via the function
#' Un_tilde()).
#'
#' @param Un_tilde Upper confidence limit of the confidence sequence by Robbins
#'   (1970).
#' @param n number of Monte-Carlo samples used.
#' @param epsilon resampling risk of the p-value estimate ptilde.
#'
#' @return Value of S (i.e. the number of successes giving rise to this upper
#'   confidence limit at this sample size and value of epsilon)
#' @keywords internal
#' @importFrom stats uniroot dbinom
#' @note For rounded values of ptilde, the ensuing value of S may be inaccurate.
invert_Un_tilde <- function(Un_tilde, n, epsilon){
  tmp <- function(S){dbinom(round(S),n,Un_tilde)- epsilon/(n+1)}
  return( max( c(round(uniroot(tmp,c(-1,round(Un_tilde*n)))$root)), 0))
}
