#' Extracts the backbone of a weighted network using results from a null model
#'
#' `backbone.extract` returns a binary or signed adjacency matrix
#'      containing the backbone that retains only the significant edges.
#'
#' @param backbone backbone: backbone S3 class object, as returned by \link{sdsm}, \link{fdsm}, or \link{hyperg}.
#' @param signed Boolean: TRUE if signed backbone is to be returned, FALSE if binary backbone is to be returned
#' @param alpha Real: Precision of significance test
#' @param fwer string: type of familywise error rate correction to be applied; c("none","bonferroni",holm"). If "holm", Holm Bonferroni Family-wise Error Rate test is used,
#'     if "bonferroni", Bonferroni Family-wise Error Rate test should be used.
#'     By default, the given 'alpha' value is used for all tests with no correction for family-wise error rates.
#' @param class string: the class of the returned backbone graph, one of c("original", "matrix", "sparseMatrix", "igraph", "network", "edgelist"), converted via \link{class.convert}. If "original", the backbone graph returned is of the same class as the data inputted in one of \link{sdsm}, \link{fdsm}, or \link{hyperg}.
#' @return backbone graph: Binary or signed backbone graph of class given in parameter `class`.
#'
#' @details The "backbone" S3 class object is composed of two matrices, a summary dataframe and (optionally, if generated by using \link{fdsm}) a 'dyad_values' vector.
#' @details The Holm Bonferroni correction was originally a port from python code written by \href{ https://github.com/saref}{Dr. Samin Aref}.
#'     The authors thank Dr. Aref greatly for his contribution to this package!
#'
#' @export
#'
#' @examples
#' probs <- sdsm(davis)
#' bb <- backbone.extract(probs, alpha = .2, signed = TRUE, fwer = "none")
backbone.extract <- function(backbone, signed = TRUE, alpha = 0.05, fwer = "none", class = "original"){

  #Argument Checks
  if ((alpha >= 1) | (alpha <= 0)) {stop("alpha must be between 0 and 1")}
  if ((class != "original")
      & (class != "matrix")
      & (class != "sparseMatrix")
      & (class != "igraph")
      & (class != "network")
      & (class != "edgelist"))
  {stop("incorrect class type, must be one of c(original, matrix, sparseMatrix, igraph, network, edgelist)")}

  #Extract components of backbone object
  positive <- as.matrix(backbone[[1]])
  negative <- as.matrix(backbone[[2]])
  summary <- backbone$summary

  if (class == "original"){
    class <- as.character(summary[1,1])
  }

  #Find p-value in the more extreme tail
  backbone <- pmin(positive,negative)  #matrix of smaller p-value from positive and negative
  diag(backbone) <- NA  #remove diagonal

  #Auxiliary values
  alpha <- alpha/2  #Alpha for each tail
  m <- sum(lower.tri(positive)==TRUE)  #Number of independent tests

  #FWER: No correction
  if(fwer=="none"){backbone <- (backbone < alpha)*1}  #significant edges

  #FWER: Bonferroni correction
  if(fwer=="bonferroni"){
    alpha <- alpha/m  #bonferroni correction
    backbone <- (backbone < alpha)*1  #significant edges
  }

  #FWER: Holm-Bonferroni correction
  if(fwer=="holm"){
    backbone[upper.tri(backbone)] <- NA  #remove upper triangle
    rank <- matrix(rank(backbone,na.last="keep",ties.method="random"),  #rank of pvalue
                   nrow=nrow(positive),
                   ncol=ncol(positive))
    holm.alphas <- alpha/(m-rank+1)  #Holm-Bonferroni step-down p-values
    backbone <- (backbone < holm.alphas)*rank  #ranks of *potentially* significant edges

    first.nonsig <- min(setdiff(c(0:m),backbone))  #find first non-significant edge rank
    backbone[backbone>=first.nonsig] <- 0  #set this and higher edge ranks as non-significant
    backbone[backbone>0] <- 1 #set remaining edge ranks as significant

    backbone[upper.tri(backbone)] <- 0  #fill upper triangle with 0s
    backbone <- backbone + t(backbone)  #fill the upper triangle with the lower triangle
  }

  #Inserting edge signs, filling diagonal
  sign <- (negative<positive)*-1  #sign of edge, if it were significant
  sign[sign==0] <- 1
  backbone <- backbone*sign  #apply appropriate sign to significant edges
  if(signed==FALSE) {backbone[backbone==-1] <- 0}  #if binary backbone requested, change -1s to 0s
  diag(backbone) <- 0  #fill diagonal with 0s

  if ((class == "dgCMatrix") | (class == "dgRMatrix") | (class == "ngRMatrix")){
    class <- "sparseMatrix"
  }

  if (class == "dgeMatrix"){
    class <- "matrix"
  }

  backbone <- class.convert(backbone, class[1])
  return(backbone[[2]])
}

