% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/si.R
\name{si}
\alias{si}
\alias{si.numeric}
\alias{si.stanreg}
\alias{si.brmsfit}
\alias{si.emmGrid}
\alias{si.data.frame}
\title{Compute Support Intervals}
\usage{
si(posterior, prior = NULL, BF = 1, verbose = TRUE, ...)

\method{si}{numeric}(posterior, prior = NULL, BF = 1, verbose = TRUE, ...)

\method{si}{stanreg}(
  posterior,
  prior = NULL,
  BF = 1,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  parameters = NULL,
  ...
)

\method{si}{brmsfit}(
  posterior,
  prior = NULL,
  BF = 1,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  parameters = NULL,
  ...
)

\method{si}{emmGrid}(posterior, prior = NULL, BF = 1, verbose = TRUE, ...)

\method{si}{data.frame}(posterior, prior = NULL, BF = 1, verbose = TRUE, ...)
}
\arguments{
\item{posterior}{A numerical vector, \code{stanreg} / \code{brmsfit} object, \code{emmGrid}
or a data frame - representing a posterior distribution(s) from (see 'Details').}

\item{prior}{An object representing a prior distribution (see 'Details').}

\item{BF}{The amount of support required to be included in the support interval.}

\item{verbose}{Toggle off warnings.}

\item{...}{Arguments passed to and from other methods.
(Can be used to pass arguments to internal \code{\link[logspline]{logspline}}.)}

\item{effects}{Should results for fixed effects, random effects or both be returned?
Only applies to mixed models. May be abbreviated.}

\item{component}{Should results for all parameters, parameters for the conditional model
or the zero-inflated part of the model be returned? May be abbreviated. Only
applies to \pkg{brms}-models.}

\item{parameters}{Regular expression pattern that describes the parameters that
should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\value{
A data frame containing the lower and upper bounds of the SI.
\cr
Note that if the level of requested support is higher than observed in the data, the
interval will be \code{[NA,NA]}.
}
\description{
A support interval contains only the values of the parameter that predict the observed data better
than average, by some degree \emph{k}; these are values of the parameter that are associated with an
updating factor greater or equal than \emph{k}. From the perspective of the Savage-Dickey Bayes factor, testing
against a point null hypothesis for any value within the support interval will yield a Bayes factor smaller
than \emph{1/k}.
\cr \cr
\strong{For more info, in particular on specifying correct priors for factors with more than 2 levels,
see \href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.}
}
\details{
This method is used to compute support intervals based on prior and posterior distributions.
For the computation of support intervals, the model priors must be proper priors (at the very least
they should be \emph{not flat}, and it is preferable that they be \emph{informative} - note
that by default, \code{brms::brm()} uses flat priors for fixed-effects; see example below).

\subsection{Choosing a value of \code{BF}}{
The choice of \code{BF} (the level of support) depends on what we want our interval to represent:
\itemize{
  \item A \code{BF} = 1 contains values whose credibility is not decreased by observing the data.
  \item A \code{BF} > 1 contains values who recived more impressive support from the data.
  \item A \code{BF} < 1 contains values whose credibility has \emph{not} been impressively decreased by observing the data.
  Testing against values outside this interval will produce a Bayes factor larger than 1/\code{BF} in support of
  the alternative. E.g., if an SI (BF = 1/3) excludes 0, the Bayes factor against the point-null will be larger than 3.
}
}
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Setting the correct \code{prior}}{

For the computation of Bayes factors, the model priors must be proper priors (at the very least
they should be \emph{not flat}, and it is preferable that they be \emph{informative}); As the priors for
the alternative get wider, the likelihood of the null value(s) increases, to the extreme that for completely
flat priors the null is infinitely more favorable than the alternative (this is called \emph{the Jeffreys-Lindley-Bartlett
paradox}). Thus, you should only ever try (or want) to compute a Bayes factor when you have an informed prior.
\cr\cr
(Note that by default, \code{brms::brm()} uses flat priors for fixed-effects; See example below.)
\cr\cr
It is important to provide the correct \code{prior} for meaningful results.
\itemize{
  \item When \code{posterior} is a numerical vector, \code{prior} should also be a numerical vector.
  \item When \code{posterior} is a \code{data.frame}, \code{prior} should also be a \code{data.frame}, with matching column order.
  \item When \code{posterior} is a \code{stanreg} or \code{brmsfit} model: \itemize{
    \item \code{prior} can be set to \code{NULL}, in which case prior samples are drawn internally.
    \item \code{prior} can also be a model equivalent to \code{posterior} but with samples from the priors \emph{only}. See \code{\link{unupdate}}.
    \item \strong{Note:} When \code{posterior} is a \code{brmsfit_multiple} model, \code{prior} \strong{must} be provided.
  }
  \item When \code{posterior} is an \code{emmGrid} object: \itemize{
    \item \code{prior} should be the \code{stanreg} or \code{brmsfit} model used to create the \code{emmGrid} objects.
    \item \code{prior} can also be an \code{emmGrid} object equivalent to \code{posterior} but created with a model of priors samples \emph{only}.
    \item \strong{Note:} When the \code{emmGrid} has undergone any transformations (\code{"log"}, \code{"response"}, etc.), or \code{regrid}ing, then \code{prior} must be an \code{emmGrid} object, as stated above.
  }
}
}

\examples{
library(bayestestR)

prior <- distribution_normal(1000, mean = 0, sd = 1)
posterior <- distribution_normal(1000, mean = .5, sd = .3)

si(posterior, prior)
\dontrun{
# rstanarm models
# ---------------
library(rstanarm)
contrasts(sleep$group) <- contr.bayes # see vingette
stan_model <- stan_lmer(extra ~ group + (1 | ID), data = sleep)
si(stan_model)
si(stan_model, BF = 3)

# emmGrid objects
# ---------------
library(emmeans)
group_diff <- pairs(emmeans(stan_model, ~group))
si(group_diff, prior = stan_model)

# brms models
# -----------
library(brms)
contrasts(sleep$group) <- contr.bayes # see vingette
my_custom_priors <-
  set_prior("student_t(3, 0, 1)", class = "b") +
  set_prior("student_t(3, 0, 1)", class = "sd", group = "ID")

brms_model <- brm(extra ~ group + (1 | ID),
  data = sleep,
  prior = my_custom_priors
)
si(brms_model)
}
}
\references{
Wagenmakers, E., Gronau, Q. F., Dablander, F., & Etz, A. (2018, November 22). The Support Interval. \doi{10.31234/osf.io/zwnxb}
}
