#' Algorithms for learning Bayesian network classifiers from data.
#' 
#' The bnclassify package implements algorithms for learning discrete Bayesian 
#' network classifiers from data. Learning can be performed on both complete and
#' incomplete data whereas prediction is much slower in the latter case and thus
#' wrapper learners are not recommended in that case.
#' 
#' Structure learning algorithms \itemize{ \item Naive Bayes \code{\link{nb}} 
#' \item Tree augmented naive Bayes \code{\link{tan_cl}} \item Backward 
#' sequential elimination and joining \code{\link{bsej}} (Pazzani, 1996) \item 
#' Forward sequential selection and joining (Pazzani, 1996) \code{\link{fssj}} 
#' \item Hill-climbing tree augmented naive Bayes \code{\link{tan_hc}} \item 
#' Hill-climbing super-parent tree augmented naive Bayes \code{\link{tan_hcsp}} 
#' }
#' 
#' Parameter estimation \itemize{ \item Attribute-weighted naive Bayes 
#' \code{\link{lpawnb}} }
#' 
#' @docType package
#' @name bnclassify
#' @importFrom stats predict as.formula complete.cases setNames 
#' @importFrom graphics plot
#' @importFrom utils combn 
#' 
#' @references Bielza C and Larranaga P (2014), Discrete Bayesian network 
#'   classifiers: A survey. \emph{ACM Computing Surveys}, \bold{47}(1), Article 
#'   5.
#'   
#'   Friedman N, Geiger D and Goldszmidt M (1997). Bayesian network classifiers.
#'   \emph{Machine Learning}, \bold{29}, pp. 131--163.
#'   
#'   Hall M (2007). A decision tree-based attribute weighting filter for naive 
#'   Bayes. \emph{Knowledge-Based Systems}, \bold{20}(2), pp. 120-126.
#'   
#'   Pazzani M (1996). Constructive induction of Cartesian product attributes. 
#'   In \emph{Proceedings of the Information, Statistics and Induction in 
#'   Science Conference (ISIS-1996)}, pp. 66-77
#'   
#'   Koegh E and Pazzani M (2002).Learning the structure of augmented Bayesian 
#'   classifiers. In \emph{International Journal on Artificial Intelligence 
#'   Tools}, \bold{11}(4), pp. 587-601.
#'   
#'   Blanco R and Inza I and Merino M and Quiroga J and Larranaga P (2005), 
#'   Feature selection in Bayesian classifiers for the prognosis of survival of 
#'   cirrhotic patients treated with TIPS. \emph{Journal of Biomedical 
#'   Informatics}, \bold{38}(5), pp. 376--388.
NULL

#' Congress Voting Data Set.
#' 
#' @source \url{http://goo.gl/GTXrCz}
#' @format A \code{data.frame} with 17 columns and 435 rows.
#' @docType data
#' @name voting
NULL

#' Car Evaluation Data Set.
#' 
#' @source \url{http://goo.gl/GTXrCz}
#' @format A \code{data.frame} with 7 columns and 1728 rows.
#' @docType data
#' @name car
NULL

#' Learns Bayesian network classifiers in a wrapper fashion.
#' 
#' Each candidate structure is evaluated according to the cross-validated
#' estimate of predictive accuracy.
#' 
#' @name wrapper
#'   
#' @inheritParams nb
#' @inheritParams cv
#' @inheritParams learn_params
#' @param epsilon A numeric. Minimum absolute improvement required to keep 
#'   searching.
#'   
#' @references Pazzani M (1996). Constructive induction of Cartesian product 
#'   attributes. In \emph{Proceedings of the Information, Statistics and 
#'   Induction in Science Conference (ISIS-1996)}, pp. 66-77
#'   
#'   Koegh E and Pazzani M (2002).Learning the structure of augmented Bayesian 
#'   classifiers. In \emph{International Journal on Artificial Intelligence 
#'   Tools}, \bold{11}(4), pp. 587-601.
NULL

#' Learns a tree augmented naive Bayes classifier (TAN).
#' 
#' Learns a one-dependence Bayesian classifier using Chow-Liu's algorithm. The 
#' structure is learned so that either likelihood, the BIC or AIC scores are 
#' maximized; the first option corresponds to the well-known tree augmented 
#' naive Bayes (Friedman et al., 1997).
#' 
#' Edges with negative BIC or AIC weights are blacklisted from the final 
#' structure. The structure thus might be a forest rather than a tree.
#' 
#' @name tan_chowliu
#'   
#' @inheritParams nb 
#' @param blacklist A character matrix. Edges that may be blacklisted from the
#'   resulting structure. Note: currently ignored.
#' @param root A character. The feature to be used as root of the augmenting 
#'   tree. Only one feature can be supplied, even in case of an augmenting 
#'   forest. This argument is optional.
#' @param score A character. The score to be maximized. \code{'loglik'},
#'   \code{'bic'}, and \code{'aic'} return the maximum likelihood, maximum BIC
#'   and maximum AIC tree/forest, respectively.
#'   
#' @references Friedman N, Geiger D and Goldszmidt M (1997). Bayesian network 
#'   classifiers. \emph{Machine Learning}, \bold{29}, pp. 131--163.
NULL

#' Learn the parameters of a Bayesian network.
#' 
#' \code{lp} estimates parameters with maximum likelihood or Bayesian 
#' estimation. \code{lpawnb} updates \code{lp} estimates according to the 
#' attribute weighted naive Bayes (AWNB) method.
#' 
#' \code{lp} learns the parameters of each \eqn{P(X_i \mid \mathbf{Pa}(X_i) = 
#' j)}{P(X[i] | Pa(X[i]) = j)} as \deqn{\theta_{ijk} = \frac{N_{ijk} + 
#' \alpha}{N_{ \cdot j \cdot } + r_i \alpha},}{\theta[ijk] = (N[ijk] + \alpha) /
#' (N[ . j . ] + r[i] \alpha),} where \eqn{N_{ijk}}{N[ijk]} is the number of 
#' instances in \code{dataset} in which \eqn{X_i = k}{X[i] = k} and 
#' \eqn{\mathbf{Pa}(X_i) = j}{Pa(X[i]) = j}, \eqn{N_{ \cdot j \cdot}}{N[ . j . 
#' ]} the number of instances in which \eqn{\mathbf{Pa}(X_i) = j}{Pa(X[i]) = j},
#' \eqn{r_i}{r[i]} is the cardinality of \eqn{X_i}{X[i]}, and all 
#' hyperparameters of the Dirichlet prior equal to \eqn{\alpha}. \eqn{\alpha = 
#' 0} corresponds to maximum likelihood estimation. Returns a uniform
#' distribution when \eqn{N_{ \cdot j \cdot } + r_i \alpha = 0}{N[ . j . ] +
#' r[i] \alpha = 0}. With partially observed data, the above amounts to
#' \emph{available case analysis}.
#' 
#' \code{lpawnb} updates the parameters learned with \code{lp} according to the 
#' AWNB, as follows, \deqn{\theta_{ijk}^{AWNB} = 
#' \frac{(\theta_{ijk})^{w_i}}{\sum_{k=1}^{r_i} 
#' (\theta_{ijk})^{w_i}},}{\theta[ijk]^(AWNB) = ((\theta[ijk])^w[i]) / 
#' (\sum[k=1]^(r[i]) (\theta[ijk])^(w[i])),} computing the weights 
#' \eqn{\mathbf{w}}{w} as \deqn{w_i = \frac{1}{M}\sum_{t=1}^M 
#' \sqrt{d_{ti}},}{w_i = (1 / M)\sum_[t=1]^M \sqrt{d[ti]},} where \eqn{M} is the
#' number of bootstrap samples from \code{dataset} and \eqn{d_{ti}}{d[ti]} the 
#' minimum testing depth of \eqn{X_i}{X[i]} in an unpruned classification tree 
#' learned from the \eqn{t}-th subsample (\eqn{d_{ti} = 0}{d[ti] = 0} if 
#' \eqn{X_i}{X_i} is omitted from \eqn{t}-th tree).
#' 
#' @name learn_params
#'   
#' @inheritParams nb
#' @param x a \code{\link{bnc_dag_object}} object. The Bayesian network 
#'   structure.
#' @param dataset The data frame from which to estimate network parameters.
#' @param smooth A numeric. The smoothing value (\eqn{\alpha}) for Bayesian 
#'   parameter estimation. Nonnegative.
#' @param trees An integer. The number (\eqn{M}) of bootstrap samples to
#'   generate.
#' @param bootstrap_size A numeric. The size of the bootstrap subsample, 
#'   relative to the size of \code{dataset} (given in [0,1]).
#' @references Mark Hall (2004). A decision tree-based attribute weighting 
#'   filter for naive Bayes. \emph{Knowledge-based Systems}, \bold{20}(2), 
#'   120-126.
NULL

#' A Bayesian network classifier structure.
#' 
#' You can query this object using the following functions.
#' 
#' @param x The bnc_dag_object
#' 
#' @name bnc_dag_object
NULL

#' A Bayesian network classifier with structure and parameters.
#' 
#' You can query this object using the following functions, in addition to all the functions that can be applied to a \code{\link{bnc_dag_object}}. 
#' 
#' @param x The bnc_bn_object
#' 
#' @name bnc_bn_object
NULL